package com.onsiteservice.miniapp.service.worker;

import com.google.common.collect.Lists;
import com.onsiteservice.constant.constant.BizConstants;
import com.onsiteservice.constant.enums.ServiceUserTypeEnum;
import com.onsiteservice.dao.common.AbstractMapper;
import com.onsiteservice.dao.mapper.service.ServiceValuatorAssignMapper;
import com.onsiteservice.dao.mapper.user.UserMapper;
import com.onsiteservice.entity.service.ServiceValuator;
import com.onsiteservice.entity.service.ServiceValuatorAssign;
import com.onsiteservice.entity.user.User;
import com.onsiteservice.miniapp.controller.worker.vo.ServiceValuatorVO;
import com.onsiteservice.util.DateUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import tk.mybatis.mapper.entity.Condition;

import javax.annotation.Resource;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author lining
 * @date 2022-07-11 16:07
 */
@Service
@Slf4j
@Transactional(rollbackFor = Exception.class)
public class ServiceValuatorService extends AbstractMapper<ServiceValuator> {


    @Resource
    private ServiceValuatorAssignMapper serviceValuatorAssignMapper;

    @Resource
    private UserMapper userMapper;


    public List<ServiceValuatorVO> all() {
        Condition c = new Condition(User.class);
        c.createCriteria().andEqualTo(BizConstants.UserConstants.ROLE_TYPE, ServiceUserTypeEnum.VALUATOR.getId());
        List<User> userList = userMapper.selectByCondition(c);

        return handle(userList);
    }


    public List<ServiceValuatorVO> selectByName(String name) {
        log.info("valuator selectByName name: {}", name);

        List<User> userList = userMapper.selectByNameLike(StringUtils.isBlank(name) ? null : name.trim());
        return handle(userList);
    }


    private List<ServiceValuatorVO> handle(List<User> userList) {
        if (CollectionUtils.isEmpty(userList)) {
            return Lists.newArrayList();
        }

        // 估价员id列表
        List<Long> valuatorIds = userList.parallelStream().map(User::getId).collect(Collectors.toList());
        // 已被指派的时间
        List<ServiceValuatorAssign> serviceValuatorAssignList = serviceValuatorAssignMapper.selectByDeletedAndIdListAndAssignTime(valuatorIds);
        Map<Long, List<ServiceValuatorAssign>> serviceValuatorAssignMap =
                serviceValuatorAssignList.parallelStream()
                        .collect(Collectors.groupingBy(ServiceValuatorAssign::getValuatorId));

        return userList.parallelStream().map(e1 -> {
            ServiceValuatorVO serviceValuatorVO = new ServiceValuatorVO();
            serviceValuatorVO.setId(e1.getId());
            serviceValuatorVO.setName(e1.getUserName());
            serviceValuatorVO.setPhone(e1.getPhone());

            // 估价员已经被指派的时间
            List<ServiceValuatorAssign> assigns = serviceValuatorAssignMap.getOrDefault(e1.getId(), Lists.newArrayList());
            String dateFormat = "yyyy-MM-dd HH:mm";
            List<String> assignTimes = assigns.parallelStream()
                    .map(assign -> DateUtils.formatDate(assign.getAssignTime(), dateFormat)).collect(Collectors.toList());

            serviceValuatorVO.setAssignTime(assignTimes);

            return serviceValuatorVO;
        }).collect(Collectors.toList());
    }


}
