package com.onsiteservice.miniapp.controller.order;

import com.onsiteservice.common.annotation.dict.Dict;
import com.onsiteservice.common.annotation.user.CurrentUserId;
import com.onsiteservice.common.order.dto.DispatchServiceOrderDTO;
import com.onsiteservice.common.order.dto.FinishServiceOrderDTO;
import com.onsiteservice.common.order.dto.SendServiceOrderDTO;
import com.onsiteservice.common.order.dto.ValuationServiceOrderDTO;
import com.onsiteservice.common.order.vo.OrderPayVO;
import com.onsiteservice.common.order.vo.ServiceOrderVO;
import com.onsiteservice.constant.constant.DictConstants;
import com.onsiteservice.constant.enums.ServiceOrderOpSourceEnum;
import com.onsiteservice.core.result.Result;
import com.onsiteservice.dao.common.page.PageInfoVO;
import com.onsiteservice.miniapp.controller.order.dto.CancelServiceOrderDTO;
import com.onsiteservice.miniapp.controller.order.dto.PageServiceOrderDTO;
import com.onsiteservice.miniapp.controller.order.dto.ReserveServiceOrderDTO;
import com.onsiteservice.miniapp.controller.order.vo.ServiceOrderDefDetailVO;
import com.onsiteservice.miniapp.service.order.ServiceOrderBizService;
import com.onsiteservice.service.order.ServiceOrderService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.NonNull;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.validation.constraints.Positive;

import static com.onsiteservice.core.result.ResultGenerator.fail;
import static com.onsiteservice.core.result.ResultGenerator.success;

/**
 * <P>订单模块</P>
 *
 * @author 李宁
 * @version v1.0
 * @since 2022/7/7 08:51
 */
@Api(tags = "订单模块")
@RestController
@RequestMapping("/order/")
@Validated
public class ServiceOrderController {


    @Resource
    private ServiceOrderBizService serviceOrderBizService;

    @Resource
    private ServiceOrderService serviceOrderService;


    @Dict(code = DictConstants.ORDER_STATUS, name = "orderStatus")
    @ApiOperation(value = "根据id查询订单详情")
    @GetMapping("get/{id}")
    public Result<ServiceOrderVO> getDetail(@ApiParam(name = "id", value = "订单id") @PathVariable @Positive Long id, @CurrentUserId Long userId) {
        return success(serviceOrderBizService.selectById(id, userId));
    }

    @Dict(code = DictConstants.ORDER_STATUS, name = "orderStatus")
    @ApiOperation(value = "工作台-分页查询订单")
    @PostMapping("page")
    public Result<PageInfoVO> getPage(@RequestBody @NonNull @Validated PageServiceOrderDTO dto, @CurrentUserId Long userId) {
        return serviceOrderBizService.getPage(dto, userId);
    }

    @Dict(code = DictConstants.ORDER_STATUS, name = "orderStatus")
    @ApiOperation(value = "我的订单-分页查询订单")
    @PostMapping("mine-page")
    public Result<PageInfoVO> getMinePage(@RequestBody @NonNull @Validated PageServiceOrderDTO dto, @CurrentUserId Long userId) {
        return serviceOrderBizService.getMinePage(dto, userId);
    }

    @ApiOperation(value = "订单预约")
    @PostMapping("reserve")
    public Result reserve(@RequestBody @NonNull @Validated ReserveServiceOrderDTO dto, @CurrentUserId Long userId) {
        return serviceOrderBizService.reserve(dto, userId, ServiceOrderOpSourceEnum.MINI_APP) == 1 ? success() : fail("已预约");
    }

    @ApiOperation(value = "订单估价")
    @PostMapping("valuation")
    public Result valuation(@RequestBody @NonNull @Validated ValuationServiceOrderDTO dto, @CurrentUserId Long userId) {
        return serviceOrderService.valuation(dto, userId, ServiceOrderOpSourceEnum.MINI_APP) == 1 ? success() : fail("估价失败");
    }

    @ApiOperation(value = "修改估价")
    @PostMapping("revaluation")
    public Result revaluation(@RequestBody @NonNull @Validated ValuationServiceOrderDTO dto, @CurrentUserId Long userId) {
        return serviceOrderService.revaluation(dto, userId, ServiceOrderOpSourceEnum.MINI_APP) == 1 ? success() : fail("修改估价失败");
    }

    /**
     * 发送订单 已估价订单才能发送 <link>ServiceOrderStatusEnum</link>
     */
    @ApiOperation(value = "发送订单")
    @PostMapping("send")
    public Result valuation(@RequestBody @NonNull @Validated SendServiceOrderDTO dto, @CurrentUserId Long userId) {
        return serviceOrderService.sendOrder(dto, userId, ServiceOrderOpSourceEnum.MINI_APP) == 1 ? success() : fail("发送失败");
    }


    @ApiOperation(value = "派单")
    @PostMapping("dispatch")
    public Result dispatch(@RequestBody @NonNull @Validated DispatchServiceOrderDTO dto, @CurrentUserId Long userId) {
        return serviceOrderService.dispatch(dto, userId, ServiceOrderOpSourceEnum.MINI_APP) == 1 ? success() : fail("派单失败");
    }

    @ApiOperation(value = "修改派单")
    @PostMapping("redispatch")
    public Result redispatch(@RequestBody @NonNull @Validated DispatchServiceOrderDTO dto, @CurrentUserId Long userId) {
        return serviceOrderService.redispatch(dto, userId, ServiceOrderOpSourceEnum.MINI_APP) == 1 ? success() : fail("修改派单失败");
    }

    /**
     * 未派单之前不能完成,已申请退款的也可以点击完成
     */
    @ApiOperation(value = "完成订单")
    @PostMapping("finish")
    public Result finish(@RequestBody @NonNull @Validated FinishServiceOrderDTO dto, @CurrentUserId Long userId) {
        return serviceOrderService.finishOrder(dto, userId, ServiceOrderOpSourceEnum.MINI_APP) == 1 ? success() : fail("取消失败");
    }

    /**
     * 订单退款 订单已付款,未派单之前可以取消 <link>ServiceOrderStatusEnum</link>
     */
//    @ApiOperation(value = "订单退款")
//    @PostMapping("refund")
//    public Result refund(@RequestBody @NonNull @Validated RefundServiceOrderDTO dto, @CurrentUserId Long userId) {
//        return serviceOrderService.refundOrder(dto, userId) == 1 ? success() : fail("退款失败");
//    }

    /**
     * 取消订单: 未支付之前可取消订单 <link>ServiceOrderStatusEnum</link>
     */
    @ApiOperation(value = "取消订单")
    @PostMapping("cancel")
    public Result cancel(@RequestBody @NonNull @Validated CancelServiceOrderDTO dto, @CurrentUserId Long userId) {
        return serviceOrderBizService.cancelOrder(dto, userId, ServiceOrderOpSourceEnum.MINI_APP) == 1 ? success() : fail("取消失败");
    }


    /**
     * 支付订单
     */
    @GetMapping("/pay/{orderId}")
    @ApiOperation(value = "发起支付")
    public Result<OrderPayVO> pay(@ApiParam(value = "订单id", required = true) @PathVariable Long orderId, @CurrentUserId Long userId, HttpServletRequest request) {
        return success(serviceOrderBizService.pay(orderId, userId, request));
    }

    /**
     * 下单默认地址
     */
    @ApiOperation(value = "订单默认信息:默认地址,温馨提示")
    @GetMapping("def-detail")
    public Result<ServiceOrderDefDetailVO> defDetail(@CurrentUserId Long userId) {
        return success(serviceOrderBizService.selectDefDetail(userId));
    }

    @ApiOperation(value = "用户协议")
    @GetMapping("agreement")
    public Result agreement() {
        return success(serviceOrderBizService.agreement());
    }

}
