package com.onsiteservice.dao.common.handler;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.onsiteservice.constant.enums.core.BaseEnum;
import com.onsiteservice.constant.enums.core.EnumUtils;
import org.apache.ibatis.type.BaseTypeHandler;
import org.apache.ibatis.type.JdbcType;
import org.apache.ibatis.type.MappedJdbcTypes;
import org.apache.ibatis.type.MappedTypes;

import java.sql.CallableStatement;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

import static org.apache.ibatis.type.JdbcType.INTEGER;
import static org.apache.ibatis.type.JdbcType.TINYINT;

/**
 * @author 潘维吉
 * @date 2019/12/23 13:45
 * @email 406798106@qq.com
 * @description 枚举类自定义类型转换
 */
@MappedTypes(BaseEnum.class)
@MappedJdbcTypes(value = {INTEGER, TINYINT})
public class EnumTypeHandler<T extends Object> extends BaseTypeHandler<T> {

    private static final ObjectMapper mapper = new ObjectMapper();
    private Class<T> clazz;

    public EnumTypeHandler(Class<T> clazz) {
        if (clazz == null) {
            throw new IllegalArgumentException("Type argument cannot be null");
        }
        this.clazz = clazz;
    }

    /**
     * 设置非空参数
     */
    @Override
    public void setNonNullParameter(PreparedStatement ps, int i, T parameter, JdbcType jdbcType) throws SQLException {
        ps.setString(i, this.toEnum(parameter).toString());
    }

    /**
     * 通过列表读取
     */
    @Override
    public T getNullableResult(ResultSet rs, String columnName) throws SQLException {
        return this.toObject(rs.getString(columnName), clazz);
    }

    /**
     * 通过下标读取
     */
    @Override
    public T getNullableResult(ResultSet rs, int columnIndex) throws SQLException {
        return this.toObject(rs.getString(columnIndex), clazz);
    }

    /**
     * 通过存储过程读取
     */
    @Override
    public T getNullableResult(CallableStatement cs, int columnIndex) throws SQLException {
        return this.toObject(cs.getString(columnIndex), clazz);
    }

    private BaseEnum toEnum(T object) {
        try {
            return EnumUtils.getEnumByCustomCode((Integer) object, BaseEnum.class);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private T toObject(String content, Class<?> clazz) {
        if (content != null && !content.isEmpty()) {
            try {
                // return (T) mapper.readValue(content, clazz);
                return EnumUtils.getEnumByCustomCode(Integer.valueOf(content), (Class<BaseEnum>) clazz);
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        } else {
            return null;
        }
    }

}
