package com.onsiteservice.dao.common;


import com.google.common.collect.Lists;
import org.apache.ibatis.exceptions.TooManyResultsException;
import org.springframework.beans.factory.annotation.Autowired;
import tk.mybatis.mapper.entity.Condition;

import java.lang.reflect.Field;
import java.lang.reflect.ParameterizedType;
import java.util.List;

/**
 * @author 潘维吉
 * @date 2018-05-19
 * 基于通用MyBatis Mapper插件的Service接口的实现 抽象类是供其他类继承的基类，抽象类不允许被实例化
 * 通用Mapper常用方法:
 * 等号的CRUD:
 * List<T> select(T record); 根据实体中的属性值进行查询，查询条件使用等号
 * T selectByPrimaryKey(Object key); 根据主键字段进行查询，方法参数必须包含完整的主键属性，查询条件使用等号
 * List<T> selectAll(); 查询全部结果，select(null)方法能达到同样的效果
 * T selectOne(T record); 根据实体中的属性进行查询，只能有一个返回值，有多个结果是抛出异常，查询条件使用等号
 * int selectCount(T record); 根据实体中的属性查询总数，查询条件使用等号
 * int insert(T record); 保存一个实体，null的属性也会保存，不会使用数据库默认值
 * int insertSelective(T record); 保存一个实体，null的属性不会保存，会使用数据库默认值
 * int updateByPrimaryKey(T record); 根据主键更新实体全部字段，null值会被更新
 * int updateByPrimaryKeySelective(T record); 根据主键更新属性不为null的值
 * int delete(T record); 根据实体属性作为条件进行删除，查询条件使用等号
 * int deleteByPrimaryKey(Object key); 根据主键字段进行删除，方法参数必须包含完整的主键属性
 * 条件的CRUD:
 * List<T> selectByCondition(Object condition); 根据Condition条件进行查询
 * int selectCountByCondition(Object condition); 根据Condition条件进行查询总数
 * int updateByCondition(@Param("record") T record, @Param("example") Object condition); 根据Condition条件更新实体record包含的全部属性，null值会被更新
 * int updateByConditionSelective(@Param("record") T record, @Param("example") Object condition); 根据Condition条件更新实体record包含的不是null的属性值
 * int deleteByCondition(Object condition); 根据Condition条件删除数据
 */
@SuppressWarnings("unchecked")
public abstract class AbstractMapper<T> {

    public static final String SAVE_EXCEPTION = "新增失败, SQL影响行数为0";
    public static final String UPDATE_EXCEPTION = "更新失败, SQL影响行数为0";

    /** 定制版MyBatis Mapper插件接口 因与通用Mapper类型一致 所以必须使用 @Autowired根据类型注入 否则会报错 */
    @Autowired
    protected Mapper<T> mapper;

    /** 当前泛型真实类型的Class */
    private Class<T> modelClass;

    public AbstractMapper() {
        ParameterizedType pt = (ParameterizedType) this.getClass().getGenericSuperclass();
        modelClass = (Class<T>) pt.getActualTypeArguments()[0];
    }

    /**
     * 通过ID查找
     *
     * @param id
     * @return Model对象
     */
    public T selectByPrimaryKey(Object id) {
        return mapper.selectByPrimaryKey(id);
    }

    /**
     * 通过多个ID查找
     * 非执行逻辑删除
     *
     * @param ids “1,2,3,4,5”
     * @return List Model对象集合
     */
    public List<T> selectByIds(String ids) {
        return mapper.selectByIds(ids);
    }

    /**
     * 通过Model中某写成员变量构造的对象查询数据库记录
     *
     * @param model
     * @return Model对象
     */
    public T selectOne(T model) {
        try {
            return mapper.selectOne(model);
        } catch (TooManyResultsException e) {
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    /**
     * 通过Model中某个成员变量名称（非数据表中column的名称）查找,value需符合unique约束
     *
     * @param propertyName 成员变量名称（非数据表中column的名称）
     * @param value        需符合unique约束
     * @return Model对象
     */
    public T selectOneByProperty(String propertyName, Object value) throws TooManyResultsException {
        try {
            T model = modelClass.getDeclaredConstructor().newInstance();
            Field field = modelClass.getDeclaredField(propertyName);
            field.setAccessible(true);
            field.set(model, value);
            return mapper.selectOne(model);
        } catch (ReflectiveOperationException e) {
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    /**
     * 根据条件查找
     *
     * @param condition
     * @return List Model对象集合
     */
    public List<T> selectByCondition(Condition condition) {
        return mapper.selectByCondition(condition);
    }

    /**
     * 获取所有记录
     *
     * @return List Model对象所有集合
     */
    public List<T> selectAll() {
        return mapper.selectAll();
    }

    /**
     * 持久化 null不会被保存
     *
     * @param model
     * @return sql执行影响的行数
     */
    public int insertSelective(T model) {
        // 保存一个实体，null的属性不会保存，会使用数据库默认值
        int affectedRows = mapper.insertSelective(model);
        if (affectedRows == 0) {
            throw new RuntimeException(SAVE_EXCEPTION);
        }
        return affectedRows;
    }

    /**
     * 持久化 null的属性会保存
     *
     * @param model
     * @return sql执行影响的行数
     */
    public int insert(T model) {
        // 保存一个实体，null的属性会保存，不会使用数据库默认值
        int affectedRows = mapper.insert(model);
        if (affectedRows == 0) {
            throw new RuntimeException(SAVE_EXCEPTION);
        }
        return affectedRows;
    }

    /**
     * 非null字段批量持久化
     *
     * @param models
     * @return sql执行影响的行数
     */
    public int insertListSelective(List<T> models) {
        // id属性并且必须为自增列 支持MySQL,H2等
        int affectedRows = mapper.insertListSelective(models);
        if (affectedRows == 0) {
            throw new RuntimeException(SAVE_EXCEPTION);
        }
        return affectedRows;
    }

    /**
     * 非null字段批量持久化 大数据量分片处理
     *
     * @param models
     * @param partition 分片数
     * @return sql执行影响的行数
     */
    public int insertListSelective(List<T> models, Integer partition) {
        // guava 实现按照固定大小分片
        List<List<T>> lists = Lists.partition(models, partition);

        int affectedRows = 0;
        for (List<T> list : lists) {
            // id属性并且必须为自增列 支持MySQL,H2等
            affectedRows = mapper.insertListSelective(list);
            if (affectedRows == 0) {
                throw new RuntimeException(SAVE_EXCEPTION);
            }
            affectedRows += affectedRows;
        }
        return affectedRows;
    }

    /**
     * 批量持久化 null会被保存
     *
     * @param models
     * @return sql执行影响的行数
     */
    public int insertList(List<T> models) {
        // id属性并且必须为自增列 支持MySQL,H2等
        int affectedRows = mapper.insertList(models);
        if (affectedRows == 0) {
            throw new RuntimeException(SAVE_EXCEPTION);
        }
        return affectedRows;
    }

    /**
     * 批量持久化 null会被保存 大数据量分片处理
     *
     * @param models
     * @param partition 分片数
     * @return sql执行影响的行数
     */
    public int insertList(List<T> models, Integer partition) {
        // guava 实现按照固定大小分片
        List<List<T>> lists = Lists.partition(models, partition);

        int affectedRows = 0;
        for (List<T> list : lists) {
            // id属性并且必须为自增列 支持MySQL,H2等
            affectedRows = mapper.insertList(list);
            if (affectedRows == 0) {
                throw new RuntimeException(SAVE_EXCEPTION);
            }
            affectedRows += affectedRows;
        }
        return affectedRows;
    }

    /**
     * 修改更新
     *
     * @param model
     * @return sql执行影响的行数
     */
    public int updateByPrimaryKeySelective(T model) {
        // 根据主键更新属性不为null的值
        int affectedRows = mapper.updateByPrimaryKeySelective(model);
        if (affectedRows == 0) {
            throw new RuntimeException(UPDATE_EXCEPTION);
        }
        return affectedRows;
    }

    /**
     * 修改全部更新 null属性会被修改
     *
     * @param model
     * @return sql执行影响的行数
     */
    public int updateByPrimaryKey(T model) {
        // 根据主键更新 括null字段
        int affectedRows = mapper.updateByPrimaryKey(model);
        if (affectedRows == 0) {
            throw new RuntimeException(UPDATE_EXCEPTION);
        }
        return affectedRows;
    }

    /**
     * 批量更新 null属性不会被修改
     *
     * @param models
     * @return sql执行影响的行数
     */
    public int updateListSelective(List<T> models) {
        int affectedRows = mapper.updateListSelective(models);
        if (affectedRows == 0) {
            throw new RuntimeException(UPDATE_EXCEPTION);
        }
        return affectedRows;
    }

    /**
     * 批量更新 null属性不会被修改 大数据量分片处理
     *
     * @param models
     * @param partition 分片数
     * @return sql执行影响的行数
     */
    public int updateListSelective(List<T> models, Integer partition) {
        // guava 实现按照固定大小分片
        List<List<T>> lists = Lists.partition(models, partition);

        int affectedRows = 0;
        for (List<T> list : lists) {
            affectedRows = mapper.updateListSelective(list);
            if (affectedRows == 0) {
                throw new RuntimeException(UPDATE_EXCEPTION);
            }
            affectedRows += affectedRows;
        }
        return affectedRows;
    }

    /**
     * 批量更新 null属性会被修改
     *
     * @param models
     * @return sql执行影响的行数
     */
    public int updateList(List<T> models) {
        int affectedRows = mapper.updateList(models);
        if (affectedRows == 0) {
            throw new RuntimeException(UPDATE_EXCEPTION);
        }
        return affectedRows;
    }

    /**
     * 批量更新 null属性会被修改 大数据量分片处理
     *
     * @param models
     * @param partition 分片数
     * @return sql执行影响的行数
     */
    public int updateList(List<T> models, Integer partition) {
        // guava 实现按照固定大小分片
        List<List<T>> lists = Lists.partition(models, partition);

        int affectedRows = 0;
        for (List<T> list : lists) {
            affectedRows = mapper.updateList(list);
            if (affectedRows == 0) {
                throw new RuntimeException(UPDATE_EXCEPTION);
            }
            affectedRows += affectedRows;
        }
        return affectedRows;
    }

    /**
     * 差异更新的方法
     * 根据两个参数 old 和 new 差异更新，当应某个字段值不同时才会更新
     *
     * @return sql执行影响的行数
     */
    public int updateByDiffer(T old, T newer) {
        int affectedRows = mapper.updateByDiffer(old, newer);
        return affectedRows;
    }

    /**
     * 自定义指定的属性(null值)会被强制更新
     *
     * @return sql执行影响的行数
     */
    public int updateByPrimaryKeySelectiveForce(T model, List<String> properties) {
        int affectedRows = mapper.updateByPrimaryKeySelectiveForce(model, properties);
        if (affectedRows == 0) {
            throw new RuntimeException(UPDATE_EXCEPTION);
        }
        return affectedRows;
    }

    /**
     * 根据Condition条件更新实体record包含的全部属性，null值会被更新
     *
     * @param record
     * @param condition
     * @return sql执行影响的行数
     */
    public int updateByCondition(T record, Condition condition) {
        int affectedRows = mapper.updateByCondition(record, condition);
        if (affectedRows == 0) {
            throw new RuntimeException(UPDATE_EXCEPTION);
        }
        return affectedRows;
    }

    /**
     * 根据Condition条件更新实体record包含的全部属性，null值不更新
     *
     * @param record
     * @param condition
     * @return sql执行影响的行数
     */
    public int updateByConditionSelective(T record, Condition condition) {
        int affectedRows = mapper.updateByConditionSelective(record, condition);
        if (affectedRows == 0) {
            throw new RuntimeException(UPDATE_EXCEPTION);
        }
        return affectedRows;
    }

    /**
     * 通过主鍵刪除
     *
     * @param id
     * @return sql执行影响的行数
     */
    public int deleteByPrimaryKey(Object id) {
        int affectedRows = mapper.deleteByPrimaryKey(id);
        return affectedRows;
    }

    /**
     * 批量刪除
     * 非执行逻辑删除
     *
     * @param ids “1,2,3,4,5”
     * @return sql执行影响的行数
     */
    public int deleteByIds(String ids) {
        int affectedRows = mapper.deleteByIds(ids);
        return affectedRows;
    }

    /**
     * 根据Condition条件删除数据
     *
     * @param condition
     * @return sql执行影响的行数
     */
    public int deleteByCondition(Condition condition) {
        int affectedRows = mapper.deleteByCondition(condition);
        return affectedRows;
    }

    /**
     * 获取总记录数
     *
     * @param model
     * @return 总记录数
     */
    public int selectCount(T model) {
        return mapper.selectCount(model);
    }

    /**
     * 根据Condition条件进行查询总记录数
     *
     * @param condition
     * @return 条件进行查询总记录数
     */
    public int selectCountByCondition(Condition condition) {
        return mapper.selectCountByCondition(condition);
    }

    /**
     * 根据主键id查询是否存在
     *
     * @param id
     * @return 是否存在
     */
    public boolean existsWithPrimaryKey(Object id) {
        return mapper.existsWithPrimaryKey(id);
    }

}
