package com.onsiteservice.core.util;

import com.alibaba.fastjson.JSON;
import com.google.common.eventbus.EventBus;
import com.onsiteservice.core.exception.ServiceException;
import com.onsiteservice.core.result.Result;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.web.method.HandlerMethod;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;

/**
 * @author 潘维吉
 * @date 2019-06-28 15:10
 * 日志处理工具
 */
@Service
@Slf4j
public class LogUtils {

    /** 自定义系统工程名称 用于区分不同项目模块  默认值空 */
    @Value("${spring.application.name:''}")
    private String projectName;
    /** EventBus事件总线  组件之间通信 */
    @Resource
    private EventBus eventBus;

    /**
     * 异常信息日志统一组合  打印和存储
     */
    public void exceptionLogs(Object handler, Exception exception, Result result,
                              String requestMethod, HttpServletRequest request, String errorMsg) {
        // 请求参数
        String requestParams = getParams(request);

        if (handler instanceof HandlerMethod) {
            HandlerMethod handlerMethod = (HandlerMethod) handler;
            String className = handlerMethod.getBean().getClass().getName();
            // 去掉类名多余的信息 方便阅读
            int markIndex = className.indexOf("$$EnhancerBySpringCGLIB");
            errorMsg = String.format(" %s方法: %s.%s, 请求参数: %s, 异常类 %s, 异常信息: %s",
                    requestMethod,
                    className.substring(0, markIndex >= 0 ? markIndex : className.length()),
                    handlerMethod.getMethod().getName(),
                    requestParams,
                    exception.getClass(),
                    errorMsg);
        } else {
            errorMsg = String.format(" %s方法, 请求参数: %s, 异常类 %s, 异常信息: %s",
                    requestMethod,
                    requestParams,
                    exception.getClass(),
                    errorMsg);
        }
        // 统一打印异常错误日志
        String stackTraceMsg = errorMsg + " 。 堆栈异常信息: " + ExceptionUtils.getStackTrace(exception);
        if (exception instanceof ServiceException) {
            log.warn(stackTraceMsg);
        } else {
            errorMsg = stackTraceMsg; // 非业务异常 存储堆栈信息
            log.error(stackTraceMsg);
        }

        // EventBus发布事件 将异常信息发送给业务模块用于存储数据库表 解耦模块之间的依赖关系
        eventBusLogs(result, errorMsg);
    }

    /**
     * EventBus发布事件 将异常信息发送给业务模块用于存储数据库表 解耦模块之间的依赖关系
     *
     * @param result
     * @param errorMsg
     */
    private void eventBusLogs(Result result, String errorMsg) {
        Map map = new HashMap(10) {{
            put("projectName", projectName);
        }};
        result.setData(map);
        result.setMsg(errorMsg);
        // 发送事件总线 用于监听者接收
        eventBus.post(result);
    }

    /**
     * 获取GET POST等请求参数
     *
     * @param request
     * @return 参数字符串
     */
    private String getParams(HttpServletRequest request) {
        //请求方法
        String requestMethod = request.getMethod();
        if ("GET".equals(requestMethod)) {
            Map<String, String[]> paramsMap = request.getParameterMap();
            Map<String, String> map = new LinkedHashMap<>();
            paramsMap.forEach((key, value) -> map.put(key, StringUtils.join(value, ",")));
            return JSON.toJSONString(map);
        } else {
/*            try {
                //安全问题 获取请求body内容数据 数据流只能读取一次 如果已读取 再次读取为空
                InputStream inputStream = request.getInputStream();
                //以utf-8格式 将输入流转换为一个字符串
                return IOUtils.toString(inputStream, "UTF-8");
            } catch (Exception e) {
                e.printStackTrace();
            }*/
            return request.getQueryString();
        }

    }
}
