package com.onsiteservice.core.config;

import com.google.common.eventbus.EventBus;
import com.google.common.eventbus.Subscribe;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import javax.annotation.Resource;
import java.lang.annotation.Annotation;
import java.lang.reflect.Method;

/**
 * @author 潘维吉
 * @date 2018-08-17
 * 注册事件消费者逻辑到事件总线
 * 检测每个bean是否是一个Guava @Subscribe定义的事件监听器bean，如果是，我们就可将该bean注册到事件总线bean eventBus上
 */
@ConditionalOnProperty(prefix = "project.event-bus", name = {"enabled"}, matchIfMissing = true)
@Configuration
public class EventBusConfig implements BeanPostProcessor {

    /** 事件总线bean由Spring IoC容器负责创建 */
    @Resource
    private EventBus eventBus;

    /**
     * 定义事件总线bean
     */
    @Bean
    public EventBus eventBus() {
        return new EventBus();
    }

    @Override
    public Object postProcessBeforeInitialization(Object bean, String beanName)
            throws BeansException {
        return bean;
    }

    /**
     * 对于每个容器执行了初始化的 bean，如果这个 bean 的某个方法注解了@Subscribe,则将该 bean 注册到事件总线
     */
    @Override
    public Object postProcessAfterInitialization(Object bean, String beanName)
            throws BeansException {
        // 对于bean中的每个方法
        Method[] methods = bean.getClass().getMethods();
        for (Method method : methods) {
            // 检查该方法的注释
            Annotation[] annotations = method.getAnnotations();
            for (Annotation annotation : annotations) {
                // 如果它包含Subscribe注释
                if (annotation.annotationType().equals(Subscribe.class)) {
                    // 如果这是一个Guava @Subscribe注解的事件监听器方法，说明所在bean实例
                    // 对应一个Guava事件监听器类，将该bean实例注册到Guava事件总线
                    eventBus.register(bean);
                    return bean;
                }
            }
        }
        return bean;
    }
}
