package com.onsiteservice.core.config;

import lombok.extern.slf4j.Slf4j;
import org.springframework.aop.interceptor.AsyncUncaughtExceptionHandler;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.annotation.Order;
import org.springframework.scheduling.annotation.AsyncConfigurer;
import org.springframework.scheduling.annotation.EnableAsync;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;

import javax.annotation.Resource;
import java.util.concurrent.Executor;
import java.util.concurrent.ThreadPoolExecutor;

/**
 * @author 潘维吉
 * @date 2019-02-20 14:56
 * 定义线程池和开启异步可用配置 修改原生 spring 异步线程池的装配
 * 提高性能往往会在主线程里面开启一个新线程去执行
 * 注解@Async声明使用异步调用  AsyncResult可以异步回调数据
 * 切记异步方法和调用方法一定要**** 写在不同的类中 ****,如果写在一个类中,是没有效果的
 * 事务里面使用异步 会开启新的线程  线程之间事务是隔离的 不相互影响
 * 多线程情况下数据库自增生成的id可能有重复问题 解决方式使用分布式主键
 */
@Order(1)
@ConditionalOnProperty(prefix = "project.async", name = {"enabled"}, matchIfMissing = true)
@Configuration
@EnableAsync // 开启线程池
@Slf4j
public class AsyncConfig implements AsyncConfigurer {

    @Value("${corePoolSize:10}")
    private int corePoolSize;
    @Value("${maxPoolSize:200}")
    private int maxPoolSize;
    @Value("${queueCapacity:500}")
    private int queueCapacity;
    @Value("${keepAliveSeconds:300}")
    private int keepAliveSeconds;

    /** 定义线程池注入 */
    @Resource
    private ThreadPoolTaskExecutor taskExecutor;

    /**
     * 自定义线程池
     */
    @Override
    public Executor getAsyncExecutor() {
        // 核心线程数
        taskExecutor.setCorePoolSize(this.corePoolSize);
        // 线程池最大线程数
        taskExecutor.setMaxPoolSize(this.maxPoolSize);
        // 线程队列容量
        taskExecutor.setQueueCapacity(this.queueCapacity);
        // 活跃时间 线程最大空闲时间
        taskExecutor.setKeepAliveSeconds(this.keepAliveSeconds);
        // 线程名字前缀
        // taskExecutor.setThreadNamePrefix("MyExecutor-");
        // setRejectedExecutionHandler：当 pool 已经达到 max size 的时候，如何处理新任务
        // CallerRunsPolicy：不在新线程中执行任务，而是由调用者所在的线程来执行
        taskExecutor.setRejectedExecutionHandler(new ThreadPoolExecutor.CallerRunsPolicy());
        // 初始化
        taskExecutor.initialize();
        return taskExecutor;
    }

    /**
     * 异步任务中异常处理
     */
    @Override
    public AsyncUncaughtExceptionHandler getAsyncUncaughtExceptionHandler() {
        return (ex, method, objects) -> {
            log.error("异步任务中异常处理信息：" + ex.getMessage() + "异常方法：" + method.getName(), ex);
        };
    }
}
