package com.onsiteservice.util;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.core.io.FileSystemResource;
import org.springframework.mail.SimpleMailMessage;
import org.springframework.mail.javamail.JavaMailSender;
import org.springframework.mail.javamail.MimeMessageHelper;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import javax.mail.internet.MimeMessage;
import java.io.File;
import java.util.Map;

/**
 * @author 潘维吉
 * @date 2020/3/3 17:45
 * @email 406798106@qq.com
 * @description 发送邮件工具类封装
 */
@ConditionalOnProperty(prefix = "spring.mail", name = {"enabled"}, matchIfMissing = false)
@Service
@Slf4j
public class MailUtils {
    /**
     * 发送邮箱  读取配置文件中的参数
     */
    @Value("${spring.mail.username}")
    private String sender;

    /** 自动注入的JavaMailSender Bean */
    @Resource
    private JavaMailSender mailSender;


    /**
     * 发送简单文本邮件  @Async异步任务提高效率
     */
    @Async
    public void send(Map<String, String> map) {
        SimpleMailMessage message = new SimpleMailMessage();
        message.setFrom(sender);//发件人
        message.setTo(map.get("email")); //收件人
        message.setSubject(map.get("title"));//主题
        message.setText(map.get("content"));//内容
        this.mailSender.send(message);
    }

    /**
     * 发送Html邮件
     */
    public void sendHtml() {
        MimeMessage message = null;
        try {
            message = this.mailSender.createMimeMessage();
            MimeMessageHelper helper = new MimeMessageHelper(message, true);
            helper.setFrom(sender);
            helper.setTo(sender);
            helper.setSubject("标题：发送Html内容");

            StringBuilder sb = new StringBuilder();
            sb.append("<h1>大标题-h1</h1>")
                    .append("<p style='color:#F00'>红色字</p>")
                    .append("<p style='text-align:right'>右对齐</p>");
            helper.setText(sb.toString(), true);
        } catch (Exception e) {
            e.printStackTrace();
        }
        this.mailSender.send(message);
    }

    /**
     * 发送带附件的邮件
     */
    public void sendAttachments() {
        MimeMessage message = null;
        try {
            message = this.mailSender.createMimeMessage();
            MimeMessageHelper helper = new MimeMessageHelper(message, true);
            helper.setFrom(sender);
            helper.setTo(sender);
            helper.setSubject("主题：带附件的邮件");
            helper.setText("带附件的邮件内容");
            //注意项目路径问题，自动补用项目路径
            FileSystemResource file = new FileSystemResource(
                    new File("src/main/resources/static/image/picture.jpg"));
            //加入邮件
            helper.addAttachment("图片.jpg", file);
        } catch (Exception e) {
            e.printStackTrace();
        }
        this.mailSender.send(message);
    }

    /**
     * 发送带静态资源的邮件
     */
    public void sendInline() {
        MimeMessage message = null;
        try {
            message = this.mailSender.createMimeMessage();
            MimeMessageHelper helper = new MimeMessageHelper(message, true);
            helper.setFrom(sender);
            helper.setTo(sender);
            helper.setSubject("主题：带静态资源的邮件");
            //第二个参数指定发送的是HTML格式,同时cid:是固定的写法
            helper.setText("<html><body>带静态资源的邮件内容 图片:<img src='cid:picture' /></body></html>", true);

            FileSystemResource file = new FileSystemResource(new File("src/main/resources/static/image/picture.jpg"));
            helper.addInline("picture", file);
        } catch (Exception e) {
            e.printStackTrace();
        }
        this.mailSender.send(message);
    }

}
