package com.onsiteservice.util;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONException;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import sun.misc.Unsafe;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @author 潘维吉
 * @date 2018-11-21 16:25
 * 公共工具类
 */
public class CommonUtils {

    /**
     * 逗号分割的数组字符串追加末尾 去掉重复添加
     */
    public static String addArrayStr(String originalStr, String addStr) {
        if (StringUtils.isBlank(addStr)) {
            return originalStr;
        } else if (StringUtils.isBlank(originalStr)) {
            return StringUtils.deleteWhitespace(addStr);
        } else {
            String[] arrays = Arrays.stream((originalStr + "," + StringUtils.deleteWhitespace(addStr)).split(","))
                    .distinct().toArray(String[]::new);
            return StringUtils.join(arrays, ",");
        }
    }

    /**
     * 逗号分割的数组字符串 部分删除
     */
    public static String deleteArrayStr(String originalStr, String deleteStr) {
        if (originalStr.equals(StringUtils.deleteWhitespace(deleteStr))) {
            return null;  // 全部删除
        } else if (StringUtils.isBlank(deleteStr)) {
            return originalStr;
        } else {
            //  差集 即要更新的数据
            return StringUtils.join(arrayDifference(
                    originalStr.split(","), deleteStr.split(",")), ",");
        }
    }

    /**
     * 根据集合获取到id字符串 以逗号,分割
     */
    public static <T> String getIdsStr(List<T> list) {
        List<Map> mapList = list.stream().map(item -> convertBeanToMap(item)
        ).collect(Collectors.toList());
        return StringUtils.join(mapList.stream().map((item) -> item.get("id")).distinct().collect(Collectors.toList()), ",");
    }

    /**
     * 字符串前后追加单引号addSingleQuote  如通用mapper条件查询使用
     */
    public static String addSingleQuote(String str) {
        return "\'" + str + "\'";
    }

    /**
     * 去掉字符串指定的前缀
     *
     * @param str    字符串名称
     * @param prefix 前缀数组
     * @return
     */
    public static String removePrefix(String str, String prefix) {
        if (StringUtils.isEmpty(str)) {
            return "";
        } else {
            if (null != prefix) {
                if (str.toLowerCase().matches("^" + prefix.toLowerCase() + ".*")) {
                    return str.substring(prefix.length());//截取前缀后面的字符串
                }
            }

            return str;
        }
    }

    /**
     * 判断一个字符串中是否全部为数字
     */
    public static boolean isNum(String str) {
        return RegexUtils.isMatch(RegexUtils.NUM, str);
    }

    public static boolean isZero(Integer value) {
        if (value == null || value == 0) {
            return true;
        }
        return false;
    }

    public static boolean isZero(Long value) {
        if (value == null || value == 0) {
            return true;
        }
        return false;
    }

    /**
     * 实体类对象转map对象方法
     *
     * @param object
     * @return
     */
    public static Map<String, Object> convertBeanToMap(Object object) {
        if (object instanceof Map) {
            return (Map) object;
        }
        Map<String, Object> map = new HashMap<>(10);
        //获得类的的属性名 数组
        Field[] fields = object.getClass().getDeclaredFields();
        try {
            for (Field field : fields) {
                field.setAccessible(true);
                String name = new String(field.getName());
                map.put(name, field.get(object));
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return map;
    }

    /**
     * map对象转实体类对象方法
     *
     * @param map
     * @return
     */
    public static Object convertMapToBean(Map<String, Object> map, Object object) {
        BeanUtils.copyProperties(map, object);
        return object;
    }

    /**
     * 数组去重
     *
     * @return
     */
    public static <T> T[] distinctArray(T[] array) {
        return (T[]) Arrays.stream(array).distinct().toArray();
    }

    /**
     * 两个数组合并
     *
     * @param first
     * @param second
     * @param <T>
     * @return
     */
    public static <T> T[] arrayConcat(T[] first, T[] second) {
        return Stream.concat(
                Stream.of(first),
                Stream.of(second)
        ).toArray(i -> (T[]) Arrays.copyOf(new Object[0], i, first.getClass()));
    }

    /**
     * 差集  返回两个数组之间的差异
     * 从first里面查找与second里不同的数组
     */
    public static <T> T[] arrayDifference(T[] first, T[] second) {
        return (T[]) Arrays.stream(first)
                .filter(item -> !Arrays.asList(second).contains(item))
                .toArray();
    }

    /**
     * 将枚举转换为 Map，其中 key 是枚举名，value 是枚举本
     *
     * @param enumClass
     * @param <E>
     * @return
     */
    public static <E extends Enum<E>> Map<String, E> getEnumMap(final Class<E> enumClass) {
        Map map = Arrays.stream(enumClass.getEnumConstants())
                .collect(Collectors.toMap(Enum::name, Function.identity(), (oldValue, newValue) -> newValue));
        map.forEach((key, value) -> {
            try {
                map.put(key, JSON.parseObject(value.toString().replaceAll("'", "\"")));
            } catch (JSONException e) {
            }
        });
        return map;
    }

    /**
     * 合并两个javabean的属性值
     *
     * @param sourceBean
     * @param targetBean
     * @param <T>
     * @return
     */
    public static <T> T mergeBeans(T sourceBean, T targetBean) {
        Class sourceBeanClass = sourceBean.getClass();
        Class targetBeanClass = targetBean.getClass();
        Field[] sourceFields = sourceBeanClass.getDeclaredFields();
        Field[] targetFields = targetBeanClass.getDeclaredFields();

        for (int i = 0; i < sourceFields.length; i++) {
            Field sourceField = sourceFields[i];
            if (Modifier.isStatic(sourceField.getModifiers())) {
                continue;
            }
            Field targetField = targetFields[i];
            if (Modifier.isStatic(targetField.getModifiers())) {
                continue;
            }
            sourceField.setAccessible(true);
            targetField.setAccessible(true);
            try {
                if ((targetField.get(targetBean) != null)) {
                    sourceField.set(sourceBean, targetField.get(targetBean));
                }
            } catch (IllegalArgumentException | IllegalAccessException e) {
                e.printStackTrace();
            }
        }
        return sourceBean;
    }

    /**
     * 以小写字符串的形式获取操作系统的名称
     *
     * @return
     */
    public static String osName() {
        return System.getProperty("os.name").toLowerCase();
    }

    /**
     * 语义化版本号x.y.z大小比较
     *
     * @param version1
     * @param version2
     * @return 0代表相等，1代表version1大于version2，-1代表version1小于version2
     */
    public static int compareVersion(String version1, String version2) {
        if (version1.equals(version2)) {
            return 0;
        }
        String[] version1Array = version1.split("\\.");
        String[] version2Array = version2.split("\\.");
        int index = 0;
        // 获取最小长度值
        int minLen = Math.min(version1Array.length, version2Array.length);
        int diff = 0;
        // 循环判断每位的大小
        while (index < minLen
                && (diff = Integer.parseInt(version1Array[index])
                - Integer.parseInt(version2Array[index])) == 0) {
            index++;
        }
        if (diff == 0) {
            // 如果位数不一致，比较多余位数
            for (int i = index; i < version1Array.length; i++) {
                if (Integer.parseInt(version1Array[i]) > 0) {
                    return 1;
                }
            }

            for (int i = index; i < version2Array.length; i++) {
                if (Integer.parseInt(version2Array[i]) > 0) {
                    return -1;
                }
            }
            return 0;
        } else {
            return diff > 0 ? 1 : -1;
        }
    }

    /**
     * 转换成0
     */
    public static BigDecimal toZero(BigDecimal value) {
        return value == null ? BigDecimal.ZERO : value;
    }

    public static Integer toZero(Integer value) {
        return value == null ? 0 : value;
    }

    public static Long toZero(Long value) {
        return value == null ? 0L : value;
    }

    /**
     * 默认除法，保留两位小数，四舍五入
     * @param dividend
     * @param divisor
     * @return
     */
    public static BigDecimal divHalfEven(BigDecimal dividend, BigDecimal divisor) {
        return dividend.divide(divisor, 2, RoundingMode.HALF_EVEN);
    }

    /**
     * 计算比率:结果在0-1之间
     *
     * @param currentValue
     * @param beforeValue
     * @return
     */
    public static BigDecimal calRadio(BigDecimal currentValue, BigDecimal beforeValue) {
        BigDecimal ratio = BigDecimal.ONE;
        if (currentValue == null || beforeValue == null) {
            return ratio;
        }
        if (beforeValue.doubleValue() > 0) {
            // 比率 = (当期 - 上期) / 上期
            ratio = divHalfEven(currentValue.subtract(beforeValue), beforeValue);
        }
        return ratio;
    }

    /**
     * 如果值为负数，则取绝对值
     * @param value
     * @return
     */
    public static BigDecimal absoluteValue(BigDecimal value) {
        value = CommonUtils.toZero(value);
        if (value.compareTo(BigDecimal.ZERO) < 0) {
            return new BigDecimal(-1).multiply(value);
        }
        return value;
    }

    /**
     * 去除IDEA控制台Warning
     */
    public static void disableWarning() {
        try {
            Field theUnsafe = Unsafe.class.getDeclaredField("theUnsafe");
            theUnsafe.setAccessible(true);
            Unsafe u = (Unsafe) theUnsafe.get(null);
            Class cls = Class.forName("jdk.internal.module.IllegalAccessLogger");
            Field logger = cls.getDeclaredField("logger");
            u.putObjectVolatile(cls, u.staticFieldOffset(logger), null);
        } catch (Exception e) {
        }
    }
}
