package com.onsiteservice.common.redis;

import com.alibaba.fastjson.support.spring.FastJsonRedisSerializer;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.data.redis.RedisAutoConfiguration;
import org.springframework.cache.annotation.EnableCaching;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.redis.cache.RedisCacheConfiguration;
import org.springframework.data.redis.cache.RedisCacheManager;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.data.redis.serializer.GenericJackson2JsonRedisSerializer;
import org.springframework.data.redis.serializer.RedisSerializationContext;
import org.springframework.data.redis.serializer.StringRedisSerializer;

import java.time.Duration;
import java.util.HashMap;
import java.util.Map;

/**
 * @author 潘维吉
 * @date 2018-05-21
 * Redis缓存配置类
 */
@ConditionalOnProperty(prefix = "project.redis", name = {"enabled"}, matchIfMissing = true)
@Configuration
@EnableCaching
@AutoConfigureAfter(RedisAutoConfiguration.class)
public class RedisConfig {

    /**
     * 注解方式的缓存的过期时间
     */
    @Value("${spring.cache.redis.time-to-live:1d}")
    private Duration timeToLive;

    /**
     * 定义 RedisTemplate ，指定序列化和反序列化的处理类
     * 使用普通的字符串保存 默认序列化的是不可读的复杂的字符串
     *
     * @param redisConnectionFactory
     * @return RedisTemplate
     * @ConditionOnMissingBean 如果已经创建了bean，则相关的初始化代码不再执行
     */
    @Bean
    @ConditionalOnMissingBean(name = "redisTemplate")
    public RedisTemplate<Object, Object> redisTemplate(RedisConnectionFactory redisConnectionFactory) {
        RedisTemplate<Object, Object> template = new RedisTemplate<>();

        // 使用fastJson序列化
        FastJsonRedisSerializer fastJsonRedisSerializer = new FastJsonRedisSerializer(Object.class);
        // value值的序列化采用fastJsonRedisSerializer
        template.setValueSerializer(fastJsonRedisSerializer);
        template.setHashValueSerializer(fastJsonRedisSerializer);
        // key的序列化采用StringRedisSerializer
        template.setKeySerializer(new StringRedisSerializer());
        template.setHashKeySerializer(new StringRedisSerializer());

        template.setConnectionFactory(redisConnectionFactory);
        return template;
    }

    /**
     * 定义 StringRedisTemplate ，指定序列化和反序列化的处理类
     * 使用普通的字符串保存 默认序列化的是不可读的复杂的字符串
     *
     * @param redisConnectionFactory
     * @return StringRedisTemplate
     * @ConditionOnMissingBean 如果已经创建了bean，则相关的初始化代码不再执行
     */
    @Bean
    @ConditionalOnMissingBean(StringRedisTemplate.class)
    public StringRedisTemplate stringRedisTemplate(RedisConnectionFactory redisConnectionFactory) {
        StringRedisTemplate template = new StringRedisTemplate();
        template.setConnectionFactory(redisConnectionFactory);
        return template;
    }

    /**
     * RedisCache配置 配置key过期时间
     *
     * @param connectionFactory
     * @return
     */
    @Bean
    public RedisCacheManager cacheManager(RedisConnectionFactory connectionFactory) {
        RedisCacheConfiguration config = RedisCacheConfiguration.defaultCacheConfig()
                .entryTtl(timeToLive) // 默认缓存时间 可根据key自定义过期时间
                .serializeKeysWith(RedisSerializationContext.SerializationPair.fromSerializer(new StringRedisSerializer()))
                .serializeValuesWith(RedisSerializationContext.SerializationPair.fromSerializer(new GenericJackson2JsonRedisSerializer()))
                .disableCachingNullValues();
        RedisCacheManager redisCacheManager = RedisCacheManager.builder(connectionFactory)
                .cacheDefaults(config)
                .withInitialCacheConfigurations(initialCacheConfigurations(config))
                .transactionAware()
                .build();

        return redisCacheManager;
    }

    /**
     * RedisCache配置 配置key过期时间等
     */
    private Map<String, RedisCacheConfiguration> initialCacheConfigurations(RedisCacheConfiguration config) {
        return new HashMap<>() {{
            // 根据业务key配置
            // put("key", config.entryTtl(Duration.ofDays(30)));
        }};
    }

}
