package com.onsiteservice.common.log;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.google.common.eventbus.Subscribe;
import com.onsiteservice.constant.constant.Constants;
import com.onsiteservice.core.exception.ServiceException;
import com.onsiteservice.core.result.Result;
import com.onsiteservice.dao.mapper.base.BaseErrorLogMapper;
import com.onsiteservice.entity.base.BaseErrorLog;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.NativeWebRequest;
import org.springframework.web.context.request.RequestAttributes;

import javax.annotation.Resource;

/**
 * @author 潘维吉
 * @date 2019-10-11
 * 定义EventBus事件消费者逻辑
 * 利用Guava的注解@Subscribe指定当某个事件发生时执行所制定的逻辑
 */

@Component
public class LogEventBusListener {

    @Value("${spring.profiles.active}")
    private String env;

    @Resource
    private NativeWebRequest nativeWebRequest;

    @Resource
    private BaseErrorLogMapper baseErrorLogMapper;

/**
     * 当所注册到的事件总线post上发生Result消息时被订阅执行
     * MvcConfig 统一异常执行 事件发送 eventBus.post(result)
     * 要用于异常日志持久化存储查询
     */
    @Subscribe
    public void listener(Result result) {
        BaseErrorLog log = new BaseErrorLog();
        //当前用户Id
        Integer userId = (Integer) nativeWebRequest.getAttribute(Constants.CURRENT_USER_ID, RequestAttributes.SCOPE_REQUEST);
        //其他json信息
        JSONObject data = ((JSONObject) JSON.toJSON(result.getData()));
        Integer code = result.getCode();
        String content = result.getMsg();

        log.setProjectName(data.getString("projectName"));
        log.setEnvironment(this.env);
        log.setErrorCode(String.valueOf(code));
        log.setContent(content);
        log.setPlatform(System.getProperty("os.name"));
        log.setCreateBy(userId == null ? null : Long.valueOf(userId));

        try {
            baseErrorLogMapper.insertSelective(log);
        } catch (Exception e) {
            throw new ServiceException("异常日志保存失败:" + result.getMsg());
        }
    }

}

