package com.onsiteservice.common.log;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.serializer.SerializerFeature;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.*;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Profile;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import java.util.LinkedHashMap;
import java.util.Map;

/**
 * @author 潘维吉
 * @date 2018-11-07
 * 核心统一日志记录切面
 * <p>
 * 日志AOP切面编程解耦
 * 模块以水平切割的方式进行分离统一处理 常用于日志、权限控制、异常处理等业务中
 * <p>
 * 编程范式主要以下几大类
 * AOP（Aspect Oriented Programming）面向切面编程
 * OOP（Object Oriented Programming）面向对象编程
 * POP（procedure oriented programming）面向过程编程
 * FP（Functional Programming）面向函数编程
 * <p>
 * AOP注解
 * @Aspect: 切面，由通知和切入点共同组成，这个注解标注在类上表示为一个切面。
 * @Joinpoint: 连接点，被AOP拦截的类或者方法，在前置通知中有介绍使用@Joinpoint获取类名、方法、请求参数。
 * Advice: 通知的几种类型
 * @Before: 前置通知，在某切入点@Pointcut之前的通知
 * @After: 后置通知，在某切入点@Pointcut之后的通知无论成功或者异常。
 * @AfterReturning: 返回后通知，方法执行return之后，可以对返回的数据做加工处理。
 * @Around: 环绕通知，在方法的调用前、后执行。
 * @AfterThrowing: 抛出异常通知，程序出错跑出异常会执行该通知方法。
 * @Pointcut: 切入点，从哪里开始。例如从某个包开始或者某个包下的某个类等。
 * <p>
 * @Profile实现多环境下的配置切换 @Profile("dev")表明只有为dev时才会实例化此类 @Profile支持类和方法 支持多参数@Profile({"dev","test"})
 */
@ConditionalOnProperty(prefix = "project.log", name = {"enabled"}, matchIfMissing = false)
@Profile("!prod")
@Component
@Aspect
@Slf4j
public class LogAspect {

    /**
     * 定义一个公共的方法，实现切入点
     * 拦截Controller下面的所有方法  任何参数(..表示拦截任何参数)
     * 以@RestController注解作为切入点  可切入其他业务模块的方法
     *
     * @within和@target针对类的注解,
     * @annotation是针对方法的注解，为自定义注解
     */
    //@Pointcut("execution(public * com.*.controller..*.*(..))")
    @Pointcut("@within(org.springframework.web.bind.annotation.RestController)")
    public void log() {
        //方法为空，因为这只是一个切入点，实现在切面编程
    }

    /**
     * 拦截方法之前的一段业务逻辑
     *
     * @param joinPoint
     */
    @Before("log()")
    public void doBefore(JoinPoint joinPoint) {
        HttpServletRequest request = getHttpServletRequest();

        Map params = new LinkedHashMap(10);
        params.put("url", request.getRequestURL()); // 获取请求的url
        params.put("method", request.getMethod()); // 获取请求的方式
        params.put("args", joinPoint.getArgs()); // 请求参数
        params.put("className", joinPoint.getSignature().getDeclaringTypeName() + "." + joinPoint.getSignature().getName()); // 获取类名和方法名
        params.put("ip", request.getRemoteAddr()); // 获取请求的ip地址

        // 输出格式化后的json字符串 IgnoreErrorGetter后FastJson会忽略有问题的get异常解析 如文件流数据 返回其它正常的解析数据
        log.info("\n\n[ {} ]接口 日志请求数据: \n{}\n", request.getRequestURI(),
                JSON.toJSONString(params, SerializerFeature.IgnoreErrorGetter));
    }

    /**
     * 环绕通知  在方法的调用前、后执行
     */
    @Around("log()")
    public Object doAround(ProceedingJoinPoint point) throws Throwable {
        HttpServletRequest request = getHttpServletRequest();
        //开始时间
        long begin = System.currentTimeMillis();
        //方法环绕proceed结果
        Object obj = point.proceed();
        //结束时间
        long end = System.currentTimeMillis();
        //时间差
        long timeDiff = (end - begin);
        String msg = "\n\n[ " + request.getRequestURI() + " ]接口 方法性能分析: \n执行总耗时 {}毫秒  来自Dream的表情: ";
        int time = 200;
        if (timeDiff < time) {
            log.info(msg + "高兴\uD83D\uDE00 \n", timeDiff);
        } else {
            log.warn(msg + "惊讶\uD83D\uDE31 总耗时超过" + time + "ms记得优化哦 \n", timeDiff);
        }
        return obj;
    }

    /**
     * 获取响应返回值  方法执行return之后
     *
     * @param object
     */
    @AfterReturning(returning = "object", pointcut = "log()")
    public void doAfterReturning(Object object) {
        HttpServletRequest request = getHttpServletRequest();
        // 会打印出一个对象，想打印出具体内容需要在定义模型处加上toString()
        log.info("\n\n[ {} ]接口 日志响应结果: \n{}\n", request.getRequestURI(), object.toString());
    }

    /**
     * 拦截方法之后的一段业务逻辑
     */
    /*
        @After("log()")
        public void doAfter() {
            log.info("doAfter");
        }
    */

    /**
     * 获取HttpServletRequest
     */
    private HttpServletRequest getHttpServletRequest() {
        ServletRequestAttributes attributes = (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
        return attributes.getRequest();
    }
}
