package com.onsiteservice.common.controller;


import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import com.github.xiaoymin.knife4j.annotations.ApiSupport;
import com.github.xiaoymin.knife4j.annotations.DynamicParameter;
import com.github.xiaoymin.knife4j.annotations.DynamicResponseParameters;
import com.onsiteservice.common.annotation.user.CurrentUserId;
import com.onsiteservice.common.controller.vo.TokenVO;
import com.onsiteservice.common.runner.InitDataRunner;
import com.onsiteservice.common.service.CommonService;
import com.onsiteservice.constant.enums.DictEnum;
import com.onsiteservice.constant.enums.FileTypeEnum;
import com.onsiteservice.core.result.Result;
import com.onsiteservice.core.security.jwt.JwtManager;
import com.onsiteservice.util.CommonUtils;
import io.swagger.annotations.*;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;
import springfox.documentation.annotations.ApiIgnore;

import javax.annotation.Resource;
import javax.validation.constraints.NotNull;
import java.util.*;

import static com.onsiteservice.core.result.ResultGenerator.success;


/**
 * @author 潘维吉
 * @date 2020/8/10 17:00
 * @email 406798106@qq.com
 * @description 通用基础业务Controller
 */
@Api(tags = "通用基础")
@ApiSupport(order = 1, author = "潘维吉") // 但是排序的最小值必须大于0
@RestController
@RequestMapping
@Validated
public class CommonController {

    /**
     * Token过期时间
     */
    @Value("${project.jwt.expires-time:-1}")
    private long expiresTime;
    @Resource
    public CommonService commonBizService;

    /**
     * 刷新登录token 用登录token换新的登录token 安全
     */
    @ApiOperationSupport(order = 1)
    @ApiOperation(value = "刷新Token", notes = "作者: 潘维吉 。备注: 每次打开app重新更新, 使用登录token请求")
    @GetMapping("/token/refresh")
    public Result<TokenVO> getRefreshToken(@CurrentUserId Long userId) {
        return success(TokenVO.builder()
                .token(JwtManager.createToken(Map.of(JwtManager.USER_ID, userId)))
                .expiresTime(expiresTime).build(), "获取刷新登录Token成功");
    }

    /**
     * 上传文件对象
     * MultipartFile[] file 方式接收多文件上传
     */
    @ApiOperationSupport(order = 2)
    @ApiOperation(value = "上传文件", notes = "作者: 潘维吉 。备注: 支持多文件")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "files", value = "文件数据", dataType = "MultipartFile", paramType = "query", allowMultiple = true, required = true)}
    )
/*    @ApiResponses(
            @ApiResponse(code = 200, message = "响应数据: {'urls': '访问urls数组(Array)'}"))*/
    @DynamicResponseParameters(name = "", properties = {
            @DynamicParameter(name = "urls", value = "访问urls", example = "[]", required = true, dataTypeClass = List.class),
    })
    @PostMapping("/upload")
    public Result upload(@RequestParam("files") MultipartFile[] files,
                         @ApiParam(value = "文件类型枚举", required = true) @RequestParam
                         @NotNull(message = "文件类型不能为空") FileTypeEnum fileType,
                         @CurrentUserId Long userId) {
        return commonBizService.uploadOss(files, fileType, userId);
    }

    /**
     * 删除文件
     */
    @ApiOperationSupport(order = 3)
    @ApiOperation(value = "删除文件", notes = "作者: 潘维吉")
    @DeleteMapping("/upload")
    public Result delete(@ApiParam(value = "文件路径", required = true) @RequestParam("path") String path) {
        return commonBizService.deleteOss(path);
    }

    /**
     * 获取字典   获取多字典code用逗号分隔
     */
    @ApiOperationSupport(order = 4)
    @ApiOperation(value = "获取字典", notes = "作者: 潘维吉 。 获取多字典code用逗号分隔")
    @GetMapping("/dict/{codes}")
    public Result getDict(@ApiParam(value = "字典编码", required = true) @PathVariable String codes) {
        Map map = new HashMap();
        Arrays.asList(StringUtils.deleteWhitespace(codes).split(",")).forEach(code -> {
            final List<Map> mapList = InitDataRunner.dictData.get(code);
            map.put(code, mapList);
        });
        return success(map, "获取字典成功");
    }

    /**
     * 获取所有区域树
     */
    @ApiOperationSupport(order = 5)
    @ApiOperation(value = "获取所有区域树", notes = "作者: 潘维吉")
    @GetMapping("/area/tree")
    public Result getAllTree() {
        return success(commonBizService.getAllTree(), "获取所有区域树");
    }

    /**
     * 获取基础Token
     */
    @ApiOperationSupport(order = 100)
    @Deprecated
    @ApiOperation(value = "获取基础Token", notes = "作者: 潘维吉 。备注: 开发阶段获取非登录状态token")
    @GetMapping("/token/base")
    public Result<TokenVO> getBaseToken() {
        return success(TokenVO.builder()
                .token(JwtManager.createToken(Map.of(JwtManager.USER_ID, 0), -1L))
                .expiresTime(-1L).build(), "获取基础Token成功");
    }

    /**
     * 根据userId获取登录Token
     */
    @Deprecated
    @ApiIgnore
    @ApiOperation(value = "根据userId获取登录Token", notes = "作者: 潘维吉 。备注: 开发使用")
    @GetMapping("/token/{userId}")
    public Result<TokenVO> getToken(@ApiParam(value = "用户Id", required = true) @PathVariable Integer userId) {
        return success(TokenVO.builder()
                .token(JwtManager.createToken(Map.of(JwtManager.USER_ID, userId)))
                .expiresTime(expiresTime).build(), "获取登录Token成功");
    }

    @Deprecated
    @ApiIgnore
    @ApiOperation(value = "获取枚举", notes = "作者: 潘维吉")
    @GetMapping("/enum/{type}")
    public Result getEnum(@ApiParam(value = "枚举类型", required = true) @PathVariable String type) {
        List list = new LinkedList();
        Map map = CommonUtils.getEnumMap(DictEnum.StatusEnum.class);
        map.forEach((key, value) -> list.add(value));
        return success(list, "获取枚举成功");
    }
}
