package com.onsiteservice.common.config;

import com.github.xiaoymin.knife4j.spring.annotations.EnableKnife4j;
import com.google.common.base.Function;
import com.google.common.base.Optional;
import com.google.common.base.Predicate;
import com.onsiteservice.common.annotation.user.CurrentUserId;
import com.onsiteservice.util.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.web.servlet.config.annotation.ResourceHandlerRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurer;
import springfox.bean.validators.configuration.BeanValidatorPluginsConfiguration;
import springfox.documentation.RequestHandler;
import springfox.documentation.builders.ApiInfoBuilder;
import springfox.documentation.builders.PathSelectors;
import springfox.documentation.service.ApiInfo;
import springfox.documentation.service.ApiKey;
import springfox.documentation.service.AuthorizationScope;
import springfox.documentation.service.SecurityReference;
import springfox.documentation.spi.DocumentationType;
import springfox.documentation.spi.service.contexts.SecurityContext;
import springfox.documentation.spring.web.plugins.Docket;
import springfox.documentation.swagger.web.UiConfiguration;
import springfox.documentation.swagger.web.UiConfigurationBuilder;
import springfox.documentation.swagger2.annotations.EnableSwagger2WebMvc;

import java.util.ArrayList;
import java.util.List;

/**
 * @author 潘维吉
 * @date 2018-05-26
 * Swagger2配置类
 * <p>
 * swagger通过注解表明该接口会生成文档，包括接口名、请求方法、参数、返回信息的等等。
 * @Api：用在请求的类上，表示对类的说明 tags="API分组标签。具有相同标签的API将会被归并在一组内展示"
 * value=" 如果tags没有定义，value将作为Api的tags使用"
 * @ApiOperation：用在请求的方法上，说明方法的用途、作用 value="说明方法的用途、作用"
 * notes="方法的备注说明"
 * @ApiParam 方法参数前添加 自动获取类型和参数名称等
 * @ApiImplicitParams：用在请求的方法上，表示一组参数说明
 * @ApiImplicitParam：用在@ApiImplicitParams注解中，指定一个请求参数的各个方面 name：参数名
 * value：参数的汉字说明、解释
 * required：参数是否必须传
 * paramType：参数放在哪个地方
 * · header --> 请求参数的获取：@RequestHeader
 * · query --> 请求参数的获取：@RequestParam
 * · path（用于restful接口）--> 请求参数的获取：@PathVariable
 * · body（不常用）
 * · form（不常用）
 * dataType：参数类型，默认String，其它值dataType="int" dataType="long"
 * defaultValue：参数的默认值
 * @ApiResponses：用在请求的方法上，表示一组响应
 * @ApiResponse：用在@ApiResponses中，一般用于表达一个错误的响应信息 code：数字，例如400
 * message：信息，例如"请求参数没填好"
 * response：抛出异常的类
 * @ApiModel：用于响应类上，表示一个返回响应数据的信息 （这种一般用在post创建的时候，使用@RequestBody这样的场景，
 * 请求参数无法使用@ApiImplicitParam注解进行描述的时候）
 * @ApiModelProperty：用在属性上，描述响应类的属性
 * @ApiIgnore()：用于类或者方法上，不被显示在页面上 <p>
 * 启动SpringBoot项目，默认UI访问 http://localhost:8080/swagger-ui.html
 * 访问接口json数据 http://localhost:8080/v2/api-docs?group=全部接口
 * knife4j增强UI访问 http://localhost:8080/doc.html
 */
//@ConditionalOnExpression("'${com.property1}'.equals('${com.property2}')")
//@ConditionalOnExpression("${swagger.enabled:true}")
@ConditionalOnProperty(prefix = "project.swagger", name = {"enabled"}, matchIfMissing = false)
// 生产环境为了安全禁用
//@Profile("!prod")
@Configuration
//@EnableSwagger2
@EnableSwagger2WebMvc
@EnableKnife4j
@Import(BeanValidatorPluginsConfiguration.class)
public class SwaggerConfig implements WebMvcConfigurer {

    @Value("${project.swagger.title:Spring Boot使用Swagger构建API文档}")
    private String title;
    @Value("${project.swagger.description:简单优雅的RESTful风格API}")
    private String description;
    @Value("${project.swagger.base-package:com.*.*.controller}")
    private String basePackage;
    @Value("${project.swagger.version:1.0.0}")
    private String version;
    @Value("${project.swagger.ignored-class-name:#{null}}")
    private List<String> ignoredClassName; // 忽略类型全包名  多包,逗号分隔 如com.**.**.CurrentUserId

    @Value("${project.swagger.head-param:#{null}}") // 增加swagger请求头，多个可用,逗号分隔
    private String headParam;

    /**
     * 将Swagger2 的swagger-ui.html加入资源路径下,否则Swagger2静态页面不能访问。要想使资源能够访问
     */
    @Override
    public void addResourceHandlers(ResourceHandlerRegistry registry) {
        registry.addResourceHandler("swagger-ui.html**", "doc.html**").
                addResourceLocations("classpath:/META-INF/resources/swagger-ui.html",
                        "classpath:/META-INF/resources/doc.html");
        registry.addResourceHandler("/webjars/**").
                addResourceLocations("classpath:/META-INF/resources/webjars/");
    }

    @Bean
    public Docket createRestApi() {
   /*   ParameterBuilder params = new ParameterBuilder();
        List<Parameter> tokenParams = new ArrayList<>();
        params.name("Authorization").description("Token令牌(Bearer前缀)").defaultValue("Bearer ")
                .modelRef(new ModelRef("string")).parameterType("header").required(false).build();
        tokenParams.add(params.build()); */
        Docket docket = new Docket(DocumentationType.SWAGGER_2)
                .groupName("全部接口")
                .apiInfo(apiInfo())
                //.pathMapping(Constants.REQUEST_PREFIX)  //设置统一的请求前缀
                .useDefaultResponseMessages(false)
                //.globalOperationParameters(tokenParams)
                .ignoredParameterTypes(CurrentUserId.class)
                .securitySchemes(securityScheme())
                .securityContexts(securityContexts())
                .select()
                //api接口包扫描路径 RequestHandlerSelectors.any()
                //.apis(RequestHandlerSelectors.basePackage(this.basePackage))
                .apis(basePackage(this.basePackage))
                //可以根据url路径设置哪些请求加入文档，忽略哪些请求  或匹配regex("/api/.*")
                .paths(PathSelectors.any())
                .build();

        // 配置忽略某些固定非用户传递的参数类型
        if (!ArrayUtils.isEmpty(ignoredClassName)) {
            ignoredClassName.forEach(item -> {
                try {
                    if (StringUtils.isNotBlank(item)) {
                        docket.ignoredParameterTypes(Class.forName(item));
                    }
                } catch (ClassNotFoundException e) {
                    e.printStackTrace();
                }
            });
        }
        return docket;
    }

    /**
     * 项目信息
     */
    private ApiInfo apiInfo() {
        return new ApiInfoBuilder()
                .title(this.title)
                .description(this.description)
                //服务条款网址
/*              .termsOfServiceUrl("https://google.com")
                .contact(new Contact("潘维吉",
                        "https://google.com",
                        "406798106@qq.com"))*/
                .version(this.version)
                .build();
    }

    /**
     * 配置认证模式
     */
    private List<ApiKey> securityScheme() {
        List<ApiKey> apiKeys = new ArrayList<>();
        ApiKey apiKey = new ApiKey("Bearer", "Authorization", "header");
        apiKeys.add(apiKey);
        if (!StringUtils.isEmpty(headParam)) {
            String[] headNames = headParam.split(",");
            for (String headName : headNames) {
                ApiKey headKey = new ApiKey(headName, headName, "header");
                apiKeys.add(headKey);
            }
        }
        return apiKeys;
    }

    /**
     * 配置认证上下文
     */
    private List<SecurityContext> securityContexts() {
        List<SecurityContext> securityContextList = new ArrayList<>();
        List<SecurityReference> securityReferenceList = new ArrayList<>();
        // defaultAuth
        securityReferenceList.add(new SecurityReference("Bearer", scopes()));
        if (!StringUtils.isEmpty(headParam)) {
            String[] headNames = headParam.split(",");
            for (String headName : headNames) {
                securityReferenceList.add(new SecurityReference(headName, scopes()));
            }
        }
        securityContextList.add(SecurityContext
                .builder()
                .securityReferences(securityReferenceList)
                .forPaths(PathSelectors.any())
                .build()
        );
        return securityContextList;
    }

    private AuthorizationScope[] scopes() {
        return new AuthorizationScope[]{
                new AuthorizationScope("global", "accessEverything")
        };
    }

    /**
     * 重写basePackage方法，使能够实现多包访问
     */
    private Predicate<RequestHandler> basePackage(final String basePackage) {
        return input -> Optional.fromNullable(input.declaringClass()).transform(handlerPackage(basePackage)).or(true);
    }

    private Function<Class<?>, Boolean> handlerPackage(final String basePackage) {
        return input -> {
            // 循环判断匹配  分隔符，逗号
            for (String strPackage : basePackage.split(",")) {
                boolean isMatch = input.getPackage().getName().startsWith(strPackage);
                if (isMatch) {
                    return true;
                }
            }
            return false;
        };
    }

    @Bean
    public UiConfiguration uiConfiguration() {
        // 默认折叠swagger ui中的Models显示 简化UI展示
        return UiConfigurationBuilder.builder().defaultModelsExpandDepth(0)
                .build();
    }
}
