package com.onsiteservice.common.annotation.user;


import com.onsiteservice.constant.constant.Constants;
import com.onsiteservice.core.exception.ServiceException;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.springframework.core.MethodParameter;
import org.springframework.stereotype.Component;
import org.springframework.web.bind.support.WebDataBinderFactory;
import org.springframework.web.context.request.NativeWebRequest;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.method.support.HandlerMethodArgumentResolver;
import org.springframework.web.method.support.ModelAndViewContainer;

/**
 * @author 潘维吉
 * @date 2018-08-20
 * 自定义解析器实现请求参数绑定方法参数
 * 增加方法注入，将含有CurrentUserId注解的方法参数注入当前请求用户
 * JwtInterceptor拦截器 已将userId存入request中
 * @CurrentUserId Long(Integer) userId 方式注入方法参数
 */
@Component
@Slf4j
public class CurrentUserIdResolver implements HandlerMethodArgumentResolver {

    /**
     * 判定是否需要处理该参数分解，返回true为需要，并会去调用下面的resolveArgument方法
     */
    @Override
    public boolean supportsParameter(MethodParameter parameter) {
        Class<?> paramType = parameter.getParameterType();
        //如果参数有@CurrentUserId注解则支持
        if ((paramType.isAssignableFrom(Long.class) || paramType.isAssignableFrom(Integer.class)) &&
                parameter.hasParameterAnnotation(CurrentUserId.class)) {
            return true;
        }
        return false;
    }

    /**
     * 真正用于处理参数分解的方法，返回的类型就是controller方法上的形参对象
     */
    @Override
    public Object resolveArgument(MethodParameter parameter, ModelAndViewContainer mavContainer,
                                  NativeWebRequest webRequest, WebDataBinderFactory binderFactory) {
        try {
            //获取jwt鉴权时存入Request的请求用户的Id
            Object currentUserId = webRequest.getAttribute(Constants.CURRENT_USER_ID,
                    RequestAttributes.SCOPE_REQUEST);
            String parameterTypeName = parameter.getParameterType().getName();
            if (ObjectUtils.isEmpty(currentUserId)) {
                // 先确认jwt拦截器 是否将数据成功放到Request中
                return null;
            } else if ("java.lang.Long".equals(parameterTypeName)) {
                return Long.valueOf(currentUserId.toString());
            } else if ("java.lang.Integer".equals(parameterTypeName)) {
                return Integer.valueOf(currentUserId.toString());
            } else if ("java.lang.String".equals(parameterTypeName)) {
                return currentUserId.toString();
            } else {
                throw new ServiceException("@CurrentUserId参数类型错误 支持Long或Integer或String");
            }
        } catch (Exception e) {
            e.printStackTrace();
            log.error("异常信息: " + e.getMessage());
            throw new RuntimeException("CurrentUserIdResolver获取当前请求用户id异常");
        }

    }
}
