package com.onsiteservice.admin.security.authentication;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.onsiteservice.constant.constant.Constants;
import com.onsiteservice.core.security.jwt.JwtManager;
import com.onsiteservice.core.security.jwt.JwtPathProperties;
import io.jsonwebtoken.Claims;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.web.filter.GenericFilterBean;

import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import java.io.IOException;
import java.util.Collections;

/**
 * JWT认证过滤器加入 spring security 管理
 */
@Slf4j
public class JwtAuthenticationFilter extends GenericFilterBean {

    @Override
    public void doFilter(ServletRequest request, ServletResponse response, FilterChain filterChain) throws IOException, ServletException {
        Authentication authentication = this.getAuthentication((HttpServletRequest) request);
        SecurityContextHolder.getContext().setAuthentication(authentication);
        filterChain.doFilter(request, response);
    }

    /**
     * 解析token 从令牌中获取数据声明 除登录外 每次请求都会调用
     * 将用户id存到request中 用于注解获取当前用户id
     *
     * @param request
     */
    private Authentication getAuthentication(HttpServletRequest request) {
        log.info("4. JwtAuthenticationFilter.getAuthentication");
        if (!JwtPathProperties.isAuth(request.getRequestURI())) { // JWT白名单放行
            // log.info("JWT白名单放行不认证: {}", request.getRequestURI());
            return null;
        } else {
            try {
                String token = request.getHeader(JwtManager.AUTHORIZATION_HEADER);
                // 返回401错误
                if (StringUtils.isBlank(token)) {
                    throw new RuntimeException();
                }
                Claims claims = JwtManager.parseToken(token.replace(JwtManager.BEARER, ""));
                if (claims != null) {
                    JSONObject jsonObject = JSON.parseObject(claims.getSubject());
                    // 如果token验证成功，将token对应的用户id
                    Object userId = jsonObject.get(JwtManager.USER_ID);
                    // 如果token验证成功，将token对应的用户id存在request中，便于之后注入
                    request.setAttribute(Constants.CURRENT_USER_ID, userId);
                    // 如果token验证成功，将token对应的用户名
                    String userName = (String) jsonObject.get(JwtManager.USER_NAME);

                    if (userName != null) {
                        return new UsernamePasswordAuthenticationToken(userName, null, Collections.emptyList());
                    }
                }
                return null;
            } catch (Exception e) {
                log.error("从令牌中获取数据声明异常: {}", e.getMessage());
                return null;
            }
        }
    }
}
