package com.onsiteservice.miniapp.service.order;

import com.github.pagehelper.PageHelper;
import com.onsiteservice.constant.constant.BizConstants;
import com.onsiteservice.constant.enums.BizCodeEnum;
import com.onsiteservice.constant.enums.ServiceOrderStatusEnum;
import com.onsiteservice.constant.enums.ServiceUserTypeEnum;
import com.onsiteservice.core.exception.ServiceException;
import com.onsiteservice.core.result.Result;
import com.onsiteservice.core.result.ResultGenerator;
import com.onsiteservice.dao.common.AbstractMapper;
import com.onsiteservice.dao.common.page.PageInfoVO;
import com.onsiteservice.dao.component.RecordComponent;
import com.onsiteservice.dao.mapper.service.*;
import com.onsiteservice.dao.mapper.user.UserMapper;
import com.onsiteservice.entity.address.ServiceAddress;
import com.onsiteservice.entity.category.ServiceCategory;
import com.onsiteservice.entity.category.ServiceSubclass;
import com.onsiteservice.entity.order.ServiceOrder;
import com.onsiteservice.entity.service.*;
import com.onsiteservice.entity.user.User;
import com.onsiteservice.miniapp.controller.order.dto.*;
import com.onsiteservice.miniapp.controller.order.vo.ServiceOrderLogVO;
import com.onsiteservice.miniapp.controller.order.vo.ServiceOrderVO;
import com.onsiteservice.miniapp.mapper.order.ServiceOrderBizMapper;
import com.onsiteservice.util.AttrCopyUtils;
import com.onsiteservice.util.RandomUtils;
import com.onsiteservice.util.aliyun.SmsUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import tk.mybatis.mapper.entity.Condition;

import javax.annotation.Resource;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

@Service
@Slf4j
@Transactional(rollbackFor = Exception.class)
public class ServiceOrderService extends AbstractMapper<ServiceOrder> {


    /**
     * 短信相关
     */
    @Value("${aliyun.sms.sign}")
    private String sign;
    @Autowired
    private SmsUtils smsUtils;


    @Resource
    private ServiceOrderBizMapper serviceOrderBizMapper;
    @Resource
    private ServiceOrderMapper serviceOrderMapper;
    @Resource
    private ServiceOrderImgMapper serviceOrderImgMapper;

    @Resource
    private ServiceAddressMapper serviceAddressMapper;

    @Resource
    private ServiceSubclassMapper serviceSubclassMapper;
    @Resource
    private ServiceCategoryMapper serviceCategoryMapper;

    @Resource
    private ServiceOrderLogMapper serviceOrderLogMapper;

    @Resource
    private ServiceWorkerMapper serviceWorkerMapper;
    @Resource
    private ServiceWorkerAssignMapper serviceWorkerAssignMapper;

    @Resource
    private ServiceValuatorMapper serviceValuatorMapper;
    @Resource
    private ServiceValuatorAssignMapper serviceValuatorAssignMapper;

    @Resource
    private UserMapper userMapper;

    @Autowired
    private RecordComponent recordComponent;


    public ServiceOrderVO selectById(Long id, Long userId) {
        log.info("order selectById id: {}, userId: {}", id, userId);

        // 订单信息
        Condition c = new Condition(ServiceOrder.class);
        c.createCriteria().andEqualTo("id", id);
        List<ServiceOrder> serviceOrderList = this.selectByCondition(c);
        if (CollectionUtils.isEmpty(serviceOrderList)) {
            throw new ServiceException(BizCodeEnum.SERVICE_ORDER_NOT_EXIST);
        }

        return buildServiceOrderVO(serviceOrderList).get(0);
    }


    public Result<PageInfoVO<ServiceOrderVO>> getPage(PageServiceOrderDTO dto, Long userId) {
        log.info("order getPage dto: {}, userId: {}", dto, userId);

        PageHelper.startPage(dto.getPage(), dto.getSize());
        List<ServiceOrder> serviceOrderList = serviceOrderMapper.selectServiceOrderPage(dto.getOrderStatus());

        return ResultGenerator.success(new PageInfoVO<>(buildServiceOrderVO(serviceOrderList)));
    }


    public int reserve(ReserveServiceOrderDTO dto, Long userId) {
        log.info("order reserve dto: {}, userId: {}", dto, userId);

        ServiceSubclass serviceSubclass = serviceSubclassMapper.selectByPrimaryKey(dto.getSubclassId());
        if (Objects.isNull(serviceSubclass)) {
            throw new ServiceException(BizCodeEnum.SERVICE_SUBCLASS_NOT_EXIST);
        }

        ServiceCategory serviceCategory = serviceCategoryMapper.selectByPrimaryKey(serviceSubclass.getCategoryId());

        ServiceAddress serviceAddress = serviceAddressMapper.selectByPrimaryKey(dto.getAddressId());
        if (Objects.isNull(serviceAddress)) {
            throw new ServiceException(BizCodeEnum.SERVICE_ADDRESS_NOT_EXIST);
        }

        ServiceOrder serviceOrder = AttrCopyUtils.copy(dto, new ServiceOrder());
        // 账户号
        serviceOrder.setAccountNo(userId);
        // 订单号
        serviceOrder.setOrderNo(RandomUtils.orderNum());
        // 客户手机号
        serviceOrder.setPhone(serviceAddress.getPhone());
        // 创建人
        serviceOrder.setCreateBy(userId);
        // 服务名
        serviceOrder.setServiceName(
                String.format(BizConstants.FormatConstants.CATEGORY_SUBCLASS_SERVICE_NAME,
                        serviceCategory.getServiceName(),
                        serviceSubclass.getServiceName()));

        int result = this.insertSelective(serviceOrder);

        recordComponent.recordServiceOrderImg(serviceOrder, dto.getUrls());

        // TODO 发短信通知所有客服

        // 记录流程
        if (result == 1) {
            recordComponent.recordProcess(serviceOrder.getId(), ServiceOrderStatusEnum.RESERVE.getStatus(), ServiceOrderStatusEnum.RESERVE.getMsg(), null, null);
        }

        return result;
    }


    /**
     * 指派估价员 管理员操作
     */
    public int valuation(ValuationServiceOrderDTO dto, Long userId) {
        log.info("order valuation dto: {}, userId: {}", dto, userId);

        User user = checkAuth(userId, BizCodeEnum.NO_AUTH_VALUATION_ORDER);

        ServiceOrder serviceOrder = checkOrder(dto.getId(), null);
        if (!serviceOrder.getOrderStatus().equals(ServiceOrderStatusEnum.RESERVE.getStatus())) {
            throw new ServiceException(BizCodeEnum.SERVICE_ORDER_CANNOT_VALUATION);
        }

        ServiceValuator serviceValuator = serviceValuatorMapper.selectByPrimaryKey(dto.getHostId());
        if (Objects.isNull(serviceValuator)) {
            throw new ServiceException(BizCodeEnum.SERVICE_VALUATOR_NOT_EXIST);
        }

        serviceValuatorAssignMapper.insertSelective(buildServiceValuatorAssign(dto, userId));

        serviceOrder.setOrderStatus(ServiceOrderStatusEnum.VALUATION.getStatus());
        serviceOrder.setModifyBy(userId);

        int result = this.updateByPrimaryKeySelective(serviceOrder);

        if (result == 1) {
            String description = String.format(ServiceOrderStatusEnum.VALUATION.getMsg(), user.getUserName(), serviceValuator.getName());
            recordComponent.recordProcess(serviceOrder.getId(), ServiceOrderStatusEnum.VALUATION.getStatus(), description, serviceValuator.getId(), dto.getRemark());
        }

        return result;
    }


    public int revaluation(ValuationServiceOrderDTO dto, Long userId) {
        log.info("order revaluation dto: {}, userId: {}", dto, userId);

        User user = checkAuth(userId, BizCodeEnum.NO_AUTH_REVALUATION_ORDER);

        ServiceOrder serviceOrder = checkOrder(dto.getId(), null);
        if (!serviceOrder.getOrderStatus().equals(ServiceOrderStatusEnum.VALUATION.getStatus())) {
            throw new ServiceException(BizCodeEnum.SERVICE_ORDER_CANNOT_REVALUATION);
        }

        ServiceValuator serviceValuator = serviceValuatorMapper.selectByPrimaryKey(dto.getHostId());
        if (Objects.isNull(serviceValuator)) {
            throw new ServiceException(BizCodeEnum.SERVICE_VALUATOR_NOT_EXIST);
        }

        disableServiceValuatorAssign(serviceOrder, userId, false);

        int result = serviceValuatorAssignMapper.insertSelective(buildServiceValuatorAssign(dto, userId));

        if (result == 1) {
            String description = String.format(ServiceOrderStatusEnum.REVALUATION.getMsg(), user.getUserName(), serviceValuator.getName());
            recordComponent.recordProcess(serviceOrder.getId(), ServiceOrderStatusEnum.REVALUATION.getStatus(), description, serviceValuator.getId(), dto.getRemark());
        }

        return result;
    }


    /**
     * 发送订单 管理员操作
     */
    public int sendOrder(SendServiceOrderDTO dto, Long userId) {
        log.info("sendOrder dto: {}, userId: {}", dto, userId);

        // 判断发送者权限 只有客服和估价员可以发送订单
        User user = userMapper.selectByPrimaryKey(userId);
        if (Objects.isNull(user) || (!user.getRoleType().equals(ServiceUserTypeEnum.ADMIN.getId()) && !user.getRoleType().equals(ServiceUserTypeEnum.VALUATOR.getId()))) {
            throw new ServiceException(BizCodeEnum.NO_AUTH_SEND_ORDER);
        }

        // 判断能发送的状态: 估价可以发送,下单可以直接发送
        ServiceOrder serviceOrder = checkOrder(dto.getId(), null);
        if (!serviceOrder.getOrderStatus().equals(ServiceOrderStatusEnum.VALUATION.getStatus()) && !serviceOrder.getOrderStatus().equals(ServiceOrderStatusEnum.RESERVE.getStatus())) {
            throw new ServiceException(BizCodeEnum.SERVICE_ORDER_CANNOT_SEND);
        }

        disableServiceValuatorAssign(serviceOrder, userId, null);

        // 估价价格
        serviceOrder.setPrice(dto.getPrice());
        serviceOrder.setOrderStatus(ServiceOrderStatusEnum.SEND.getStatus());
        serviceOrder.setModifyBy(userId);

        int result = serviceOrderMapper.updateByPrimaryKeySelective(serviceOrder);

        try {
            // 发短信
            log.info("sendOrder send msg to phone: {}", serviceOrder.getPhone());
            // TODO 短信通知客户完成支付
            smsUtils.send(sign, "", "", serviceOrder.getPhone());
        } catch (Exception e) {
            log.error("sendOrder error, ", e);
        }

        // TODO 小程序通知客户完成支付

        if (result == 1) {
            boolean isAdmin = user.getRoleType().equals(ServiceUserTypeEnum.ADMIN.getId());
            String description = String.format(ServiceOrderStatusEnum.SEND.getMsg(), (isAdmin ? ServiceUserTypeEnum.ADMIN.getName() : ServiceUserTypeEnum.VALUATOR.getName()) + user.getUserName());
            recordComponent.recordProcess(serviceOrder.getId(), ServiceOrderStatusEnum.SEND.getStatus(), description, null, null);
        }

        return result;
    }


    public int pay(PayServiceOrderDTO dto, Long userId) {
        log.info("order pay dto: {}, userId: {}", dto, userId);

        // 判断能发送的状态
        ServiceOrder serviceOrder = checkOrder(dto.getId(), userId);
        if (!serviceOrder.getOrderStatus().equals(ServiceOrderStatusEnum.SEND.getStatus())) {
            throw new ServiceException(BizCodeEnum.SERVICE_ORDER_CANNOT_PAY);
        }

        serviceOrder.setOrderStatus(ServiceOrderStatusEnum.PAY.getStatus());
        serviceOrder.setModifyBy(userId);

        int result = serviceOrderMapper.updateByPrimaryKeySelective(serviceOrder);

        // TODO 调用微信支付接口

        if (result == 1) {
            recordComponent.recordProcess(serviceOrder.getId(), ServiceOrderStatusEnum.PAY.getStatus(), ServiceOrderStatusEnum.PAY.getMsg(), null, null);
        }

        return result;
    }


    public int dispatch(DispatchServiceOrderDTO dto, Long userId) {
        log.info("order dispatch dto: {}, userId: {}", dto, userId);

        User user = checkAuth(userId, BizCodeEnum.NO_AUTH_DISPATCH_ORDER);

        ServiceOrder serviceOrder = checkOrder(dto.getId(), null);
        if (!serviceOrder.getOrderStatus().equals(ServiceOrderStatusEnum.PAY.getStatus())) {
            throw new ServiceException(BizCodeEnum.SERVICE_ORDER_CANNOT_DISPATCH);
        }

        ServiceWorker serviceWorker = serviceWorkerMapper.selectByPrimaryKey(dto.getHostId());
        if (Objects.isNull(serviceWorker)) {
            throw new ServiceException(BizCodeEnum.SERVICE_WORKER_NOT_EXIST);
        }

        serviceWorkerAssignMapper.insertSelective(buildServiceWorkerAssign(dto, userId));

        serviceOrder.setOrderStatus(ServiceOrderStatusEnum.DISPATCH.getStatus());
        serviceOrder.setModifyBy(userId);

        int result = serviceOrderMapper.updateByPrimaryKeySelective(serviceOrder);

        // TODO 短信 "您的订单已为您指派维修工,请保持手机畅通!"

        // TODO 小程序通知 您的订单已为您指派维修工~~~

        if (result == 1) {
            String description = String.format(ServiceOrderStatusEnum.DISPATCH.getMsg(), user.getUserName(), serviceWorker.getName());
            recordComponent.recordProcess(serviceOrder.getId(), ServiceOrderStatusEnum.DISPATCH.getStatus(), description, serviceWorker.getId(), dto.getRemark());
        }

        return result;
    }


    public int redispatch(DispatchServiceOrderDTO dto, Long userId) {
        log.info("order redispatch dto: {}, userId: {}", dto, userId);

        User user = checkAuth(userId, BizCodeEnum.NO_AUTH_REDISPATCH_ORDER);

        ServiceOrder serviceOrder = checkOrder(dto.getId(), null);
        if (!serviceOrder.getOrderStatus().equals(ServiceOrderStatusEnum.DISPATCH.getStatus())) {
            throw new ServiceException(BizCodeEnum.SERVICE_ORDER_CANNOT_REDISPATCH);
        }

        ServiceWorker serviceWorker = serviceWorkerMapper.selectByPrimaryKey(dto.getHostId());
        if (Objects.isNull(serviceWorker)) {
            throw new ServiceException(BizCodeEnum.SERVICE_WORKER_NOT_EXIST);
        }

        disableServiceWorkerAssign(serviceOrder, userId, false);

        int result = serviceWorkerAssignMapper.insertSelective(buildServiceWorkerAssign(dto, userId));

        // TODO 短信 "您的订单已为您重新指派维修工,请保持手机畅通!"

        // TODO 小程序通知 您的订单已为您重新指派维修工~~~

        if (result == 1) {
            String description = String.format(ServiceOrderStatusEnum.REDISPATCH.getMsg(), user.getUserName(), serviceWorker.getName());
            recordComponent.recordProcess(serviceOrder.getId(), ServiceOrderStatusEnum.REDISPATCH.getStatus(), description, serviceWorker.getId(), dto.getRemark());
        }

        return result;
    }


    /**
     * 完成
     */
    public int finishOrder(FinishServiceOrderDTO dto, Long userId) {
        log.info("finishOrder dto: {}, userId: {}", dto, userId);

        ServiceOrder serviceOrder = checkOrder(dto.getId(), null);
        if (!serviceOrder.getOrderStatus().equals(ServiceOrderStatusEnum.DISPATCH.getStatus())) {
            throw new ServiceException(BizCodeEnum.SERVICE_ORDER_CANNOT_FINISH);
        }

        User user = userMapper.selectByPrimaryKey(userId);
        // 既不是客户 又不是客服
        boolean isUser = serviceOrder.getAccountNo().equals(userId);
        boolean isAdmin = !Objects.isNull(user) && user.getRoleType().equals(ServiceUserTypeEnum.ADMIN.getId());
        if (!isUser && !isAdmin) {
            throw new ServiceException(BizCodeEnum.NO_AUTH_FINISH_ORDER);
        }

        disableServiceWorkerAssign(serviceOrder, userId, null);

        serviceOrder.setOrderStatus(ServiceOrderStatusEnum.FINISH.getStatus());
        serviceOrder.setModifyBy(userId);

        int result = serviceOrderMapper.updateByPrimaryKeySelective(serviceOrder);

        if (result == 1) {
            String description = String.format(ServiceOrderStatusEnum.FINISH.getMsg(), isUser ? ServiceUserTypeEnum.USER.getName() : ServiceUserTypeEnum.ADMIN.getName() + user.getUserName());
            recordComponent.recordProcess(serviceOrder.getId(), ServiceOrderStatusEnum.FINISH.getStatus(), description, null, null);
        }

        return result;
    }


    public int cancelOrder(CancelServiceOrderDTO dto, Long userId) {
        log.info("cancelOrder dto: {}, userId: {}", dto, userId);

        ServiceOrder serviceOrder = checkOrder(dto.getId(), userId);
        if (serviceOrder.getOrderStatus() >= ServiceOrderStatusEnum.PAY.getStatus()) {
            throw new ServiceException(BizCodeEnum.SERVICE_ORDER_CANNOT_CANCEL);
        }

        disableServiceValuatorAssign(serviceOrder, userId, null);

        serviceOrder.setOrderStatus(ServiceOrderStatusEnum.CANCEL.getStatus());
        serviceOrder.setModifyBy(userId);

        int result = serviceOrderMapper.updateByPrimaryKeySelective(serviceOrder);

        if (result == 1) {
            recordComponent.recordProcess(serviceOrder.getId(), ServiceOrderStatusEnum.CANCEL.getStatus(), ServiceOrderStatusEnum.CANCEL.getMsg(), null, null);
        }

        return result;
    }


    public List<ServiceOrderLogVO> process(Long id, Long userId) {
        log.info("order process id: {}, userId: {}", id, userId);

        Condition c = new Condition(ServiceOrderLog.class);
        c.createCriteria().andEqualTo(BizConstants.OrderConstants.ORDER_ID, id);
        List<ServiceOrderLog> serviceOrderLogList = serviceOrderLogMapper.selectByCondition(c);
        return serviceOrderLogList.parallelStream().map(e -> AttrCopyUtils.copy(e, new ServiceOrderLogVO())).sorted(Comparator.comparing(ServiceOrderLogVO::getCreateTime).reversed()).collect(Collectors.toList());
    }


    /**
     * 未派单订单 用户可实时退款
     * 已派单订单 需要管理员审批
     */
//    public int refundOrder(RefundServiceOrderDTO dto, Long userId) {
//        log.info("refundOrder dto: {}, userId: {}", dto, userId);
//
//        ServiceOrder serviceOrder = checkOrder(dto.getId(), userId);
//        if (!serviceOrder.getOrderStatus().equals(ServiceOrderStatusEnum.PAY.getStatus())) {
//            throw new ServiceException(BizCodeEnum.SERVICE_ORDER_CANNOT_REFUND);
//        }
//
//        // 已申请退款
//        serviceOrder.setOrderStatus(ServiceOrderStatusEnum.REFUND_APPLY.getStatus());
//        int result = serviceOrderMapper.updateByPrimaryKeySelective(serviceOrder);
//
//
//        try {
//            //  调用微信接口退款 ??
//
//            //  通知web
//        } catch (Exception e) {
//
//        }
//
//
//        return result;
//    }
    private ServiceOrder checkOrder(Long id, Long userId) {
        Condition c = new Condition(ServiceOrder.class);

        c.createCriteria().andEqualTo("id", id);
        if (userId != null) {
            c.createCriteria().andEqualTo(BizConstants.UserConstants.ACCOUNT_NO, userId);
        }

        List<ServiceOrder> serviceOrderList = serviceOrderMapper.selectByCondition(c);
        if (CollectionUtils.isEmpty(serviceOrderList)) {
            throw new ServiceException(BizCodeEnum.SERVICE_ORDER_NOT_EXIST);
        }

        return serviceOrderList.get(0);
    }


    private User checkAuth(Long userId, BizCodeEnum biz) {
        User user = userMapper.selectByPrimaryKey(userId);
        if (Objects.isNull(user) || !user.getRoleType().equals(ServiceUserTypeEnum.ADMIN.getId())) {
            throw new ServiceException(biz);
        }

        return user;
    }


    private ServiceValuatorAssign buildServiceValuatorAssign(ValuationServiceOrderDTO dto, Long userId) {
        ServiceValuatorAssign serviceValuatorAssign = new ServiceValuatorAssign();
        // 估价员id
        serviceValuatorAssign.setValuatorId(dto.getHostId());
        // 订单id
        serviceValuatorAssign.setOrderId(dto.getId());
        // 估价员上门时间
        serviceValuatorAssign.setAssignTime(dto.getExpectArrivalTime());
        // 创建人
        serviceValuatorAssign.setCreateBy(userId);

        return serviceValuatorAssign;
    }


    /**
     * 估价员上门时间失效
     *
     * @param serviceOrder  订单
     * @param userId        用户id
     * @param finalExecuted 逻辑上谁最有可能是最终的上门人员
     */
    public void disableServiceValuatorAssign(ServiceOrder serviceOrder, Long userId, Boolean finalExecuted) {
        log.info("disableServiceValuatorAssign serviceOrder: {}, userId: {}, finalExecuted: {}", serviceOrder, userId, finalExecuted);

        ServiceValuatorAssign serviceValuatorAssign = new ServiceValuatorAssign();
        serviceValuatorAssign.setModifyBy(userId);
        serviceValuatorAssign.setDeleted(true);
        if (finalExecuted != null) {
            serviceValuatorAssign.setFinalExecuted(finalExecuted);
        }

        Condition c = new Condition(ServiceValuatorAssign.class);
        c.createCriteria().andEqualTo(BizConstants.OrderConstants.ORDER_ID, serviceOrder.getId());

        serviceValuatorAssignMapper.updateByConditionSelective(serviceValuatorAssign, c);
    }


    private ServiceWorkerAssign buildServiceWorkerAssign(DispatchServiceOrderDTO dto, Long userId) {
        ServiceWorkerAssign serviceWorkerAssign = new ServiceWorkerAssign();
        // 维修工id
        serviceWorkerAssign.setWorkerId(dto.getHostId());
        // 订单id
        serviceWorkerAssign.setOrderId(dto.getId());
        // 维修工上门时间
        serviceWorkerAssign.setAssignTime(dto.getExpectArrivalTime());
        // 创建人
        serviceWorkerAssign.setCreateBy(userId);

        return serviceWorkerAssign;
    }


    /**
     * 维修工上门时间失效
     *
     * @param serviceOrder 订单
     * @param userId       用户id
     */
    public void disableServiceWorkerAssign(ServiceOrder serviceOrder, Long userId, Boolean finalExecuted) {
        log.info("disableServiceWorkerAssign serviceOrder: {}, userId: {}, finalExecuted: {}", serviceOrder, userId, finalExecuted);

        ServiceWorkerAssign serviceWorkerAssign = new ServiceWorkerAssign();
        serviceWorkerAssign.setModifyBy(userId);
        serviceWorkerAssign.setDeleted(true);
        if (finalExecuted != null) {
            serviceWorkerAssign.setFinalExecuted(finalExecuted);
        }

        Condition c = new Condition(ServiceWorkerAssign.class);
        c.createCriteria().andEqualTo(BizConstants.OrderConstants.ORDER_ID, serviceOrder.getId());

        serviceWorkerAssignMapper.updateByConditionSelective(serviceWorkerAssign, c);
    }


    private List<ServiceOrderVO> buildServiceOrderVO(List<ServiceOrder> serviceOrderList) {
        // 地址信息
        List<Long> addressIdList = serviceOrderList.parallelStream().map(ServiceOrder::getAddressId).collect(Collectors.toList());
        List<ServiceAddress> serviceAddressList = serviceAddressMapper.selectByIdList(addressIdList);
        Map<Long, ServiceAddress> serviceAddressMap = serviceAddressList.parallelStream().collect(Collectors.toMap(ServiceAddress::getId, Function.identity()));

        // 图片信息
        List<Long> orderIdList = serviceOrderList.parallelStream().map(ServiceOrder::getId).collect(Collectors.toList());
        List<ServiceOrderImg> serviceOrderImgList = serviceOrderImgMapper.selectByOrderIdList(orderIdList);
        Map<Long, List<ServiceOrderImg>> serviceOrderImgMap = serviceOrderImgList.parallelStream().collect(Collectors.groupingBy(ServiceOrderImg::getOrderId));

        return serviceOrderList.parallelStream().map(e -> {
            ServiceOrderVO serviceOrderVO = AttrCopyUtils.copy(e, new ServiceOrderVO());
            ServiceAddress sa = serviceAddressMap.getOrDefault(serviceOrderVO.getAddressId(), new ServiceAddress());
            serviceOrderVO.setName(sa.getName());
            serviceOrderVO.setAddress(sa.getAddress());
            serviceOrderVO.setUrls(serviceOrderImgMap.getOrDefault(serviceOrderVO.getId(), new ArrayList<>()).parallelStream()
                    .map(ServiceOrderImg::getUrl).collect(Collectors.toList()));

            return serviceOrderVO;
        }).collect(Collectors.toList());
    }


}
