package com.onsiteservice.miniapp.service.address;

import com.github.pagehelper.PageHelper;
import com.onsiteservice.constant.constant.BizConstants;
import com.onsiteservice.constant.enums.BizCodeEnum;
import com.onsiteservice.core.exception.ServiceException;
import com.onsiteservice.core.result.Result;
import com.onsiteservice.core.result.ResultGenerator;
import com.onsiteservice.dao.common.AbstractMapper;
import com.onsiteservice.dao.common.page.PageInfoVO;
import com.onsiteservice.dao.mapper.service.ServiceAreaMapper;
import com.onsiteservice.entity.address.ServiceAddress;
import com.onsiteservice.entity.area.ServiceArea;
import com.onsiteservice.miniapp.controller.address.dto.PageServiceAddressDTO;
import com.onsiteservice.miniapp.controller.address.dto.SaveServiceAddressDTO;
import com.onsiteservice.miniapp.controller.address.dto.UpdateServiceAddressDTO;
import com.onsiteservice.miniapp.controller.address.vo.ServiceAddressVO;
import com.onsiteservice.miniapp.mapper.address.ServiceAddressBizMapper;
import com.onsiteservice.util.AttrCopyUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import tk.mybatis.mapper.entity.Condition;

import javax.annotation.Resource;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

@Service
@Slf4j
@Transactional(rollbackFor = Exception.class)
public class ServiceAddressService extends AbstractMapper<ServiceAddress> {


    @Resource
    private ServiceAddressBizMapper serviceAddressBizMapper;

    @Resource
    private ServiceAreaMapper serviceAreaMapper;


    /**
     * 分页查询列表
     */
    public Result<PageInfoVO<ServiceAddressVO>> getPage(PageServiceAddressDTO dto, Long userId) {
        log.info("address getPage dto: {}, uid: {}", dto, userId);

        PageHelper.startPage(dto.getPage(), dto.getSize());
        List<ServiceAddress> serviceAddressList = serviceAddressBizMapper.selectServiceAddressPage(dto, userId);
        List<ServiceAddressVO> serviceAddressVOList = serviceAddressList.parallelStream().map(e -> AttrCopyUtils.copy(e, new ServiceAddressVO())).collect(Collectors.toList());
        return ResultGenerator.success(new PageInfoVO<>(serviceAddressVOList));
    }


    public ServiceAddressVO selectById(Long id, Long userId) {
        log.info("address selectById id: {}, userId: {}", id, userId);

        Condition c = new Condition(ServiceAddress.class);
        c.createCriteria().andEqualTo("id", id).andEqualTo(BizConstants.UserConstants.ACCOUNT_NO, userId);

        List<ServiceAddress> serviceAddressList = this.selectByCondition(c);
        if (CollectionUtils.isEmpty(serviceAddressList)) {
            throw new ServiceException(BizCodeEnum.SERVICE_ADDRESS_NOT_EXIST);
        }

        return AttrCopyUtils.copy(serviceAddressList.get(0), new ServiceAddressVO());
    }


    public List<ServiceAddressVO> selectByUserId(Long userId) {
        Condition condition = new Condition(ServiceAddress.class);
        condition.createCriteria().andEqualTo(BizConstants.UserConstants.ACCOUNT_NO, userId);
        List<ServiceAddress> serviceAddresses = this.selectByCondition(condition);

        return serviceAddresses.parallelStream()
                .map(e -> AttrCopyUtils.copy(e, new ServiceAddressVO()))
                .collect(Collectors.toList());
    }


    public int save(SaveServiceAddressDTO dto, Long userId) {
        log.info("address save dto: {}, userId: {}", dto, userId);

        checkServiceArea(dto.getAreaId());

        ServiceAddress serviceAddress = AttrCopyUtils.copy(dto, new ServiceAddress());
        serviceAddress.setAccountNo(userId);
        // 有默认收货地址的话,置为非默认
        checkAndCancelDefaultServiceAddress(serviceAddress, userId);

        return this.insertSelective(serviceAddress);
    }


    public int update(UpdateServiceAddressDTO dto, Long userId) {
        log.info("address update dto: {}, userId: {}", dto, userId);

        checkServiceArea(dto.getAreaId());

        ServiceAddress serviceAddress = AttrCopyUtils.copy(dto, new ServiceAddress());
        serviceAddress.setAccountNo(userId);
        checkAndCancelDefaultServiceAddress(serviceAddress, userId);

        Condition c = new Condition(ServiceAddress.class);
        c.createCriteria().andEqualTo("id", dto.getId()).andEqualTo(BizConstants.UserConstants.ACCOUNT_NO, userId);
        return this.updateByConditionSelective(serviceAddress, c);
    }


    public int delete(Long id, Long userId) {
        log.info("address delete id: {}, userId: {}", id, userId);

        Condition c = new Condition(ServiceAddress.class);
        // 删除条件里加上账户号防越权攻击
        c.createCriteria().andEqualTo("id", id).andEqualTo(BizConstants.UserConstants.ACCOUNT_NO, userId);
        return this.deleteByCondition(c);
    }


    /**
     * 判断区域存在
     *
     * @param areaId 区域id
     */
    private void checkServiceArea(long areaId) {
        ServiceArea serviceArea = serviceAreaMapper.selectByPrimaryKey(areaId);
        if (Objects.isNull(serviceArea)) {
            throw new ServiceException(BizCodeEnum.SERVICE_AREA_NOT_EXIST);
        }
    }


    /**
     * 判断有没有默认收货地址,有的话取消
     *
     * @param serviceAddress 服务地址DO
     * @param userId         用户id
     */
    private void checkAndCancelDefaultServiceAddress(ServiceAddress serviceAddress, Long userId) {
        if (serviceAddress.getDef()) {
            Condition c = new Condition(ServiceAddress.class);
            c.createCriteria().andEqualTo(BizConstants.UserConstants.ACCOUNT_NO, userId)
                    .andEqualTo("def", true);
            List<ServiceAddress> defaultServiceAddressList = this.selectByCondition(c);
            if (!CollectionUtils.isEmpty(defaultServiceAddressList)) {
                defaultServiceAddressList.forEach(defaultServiceAddress -> {
                    defaultServiceAddress.setDef(false);
                    this.updateByPrimaryKeySelective(defaultServiceAddress);
                });
            }
        }
    }



}
