package com.onsiteservice.dao.common.schedule;

import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.context.event.EventListener;
import org.springframework.scheduling.TaskScheduler;
import org.springframework.scheduling.support.CronTrigger;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.*;
import java.util.concurrent.ScheduledFuture;


/**
 * @author 潘维吉
 * @date 2019/12/9 10:40
 * @email 406798106@qq.com
 * @description 实现动态创建更新取消定时任务
 */
@ConditionalOnProperty(prefix = "project.schedule.dynamic", name = {"enabled"}, matchIfMissing = false)
@Service
@Slf4j
public class DynamicScheduleService {

    /** 所有的定时jobs */
    Map<Integer, ScheduledFuture<?>> taskJobs = new HashMap<>();
    /** 自定义多线程任务 */
    @Resource(name = "myTaskScheduler")
    private TaskScheduler taskScheduler;

    /**
     * 动态添加任务
     *
     * @param id   任务唯一标识
     * @param task 任务线程
     * @param cron cron 表达式
     */
    public boolean add(int id, Runnable task, String cron) {
        ScheduledFuture<?> scheduledTask = taskJobs.get(id);
        if (scheduledTask == null) {
            scheduledTask = taskScheduler.schedule(
                    task, new CronTrigger(cron, TimeZone.getTimeZone(TimeZone.getDefault().getID())));
            taskJobs.put(id, scheduledTask);
            return true;
        } else {
            return false;
        }
    }

    /**
     * 更新定时任务
     *
     * @param id   任务唯一标识
     * @param task 任务线程
     * @param cron cron 表达式
     */
    public boolean update(int id, Runnable task, String cron) {
        if (remove(id)) {
            return add(id, task, cron);
        }
        return false;
    }

    /**
     * 删除定时任务
     *
     * @param id 任务唯一标识
     */
    public boolean remove(int id) {
        ScheduledFuture<?> scheduledTask = taskJobs.get(id);
        if (scheduledTask != null) {
            scheduledTask.cancel(true);
            taskJobs.put(id, null);
            return true;
        }
        return false;
    }

    /**
     * 上下文刷新事件监听器  即服务器重新启动
     */
    @EventListener({ContextRefreshedEvent.class})
    void contextRefreshedEvent() {
        // log.info("从DB获取所有任务，并在上下文重新启动时重新安排它们");
        /**
         * 测试数据，实际可从数据库获取
         */
        List<Task> tasks = Arrays.asList(
                new Task(1, () -> log.info("动态定时执行任务1"), "*/10 * * * * *"),
                new Task(2, () -> log.info("动态定时执行任务2"), "*/20 * * * * *"),
                new Task(3, () -> log.info("动态定时执行任务3"), "*/30 * * * * *")
        );

        tasks.forEach(item -> {
            //  add(item.getId(), item.getRunnable(), item.cron);
        });
    }


    @Getter
    @Setter
    @AllArgsConstructor
    static class Task {
        /**
         * 主键ID
         */
        private Integer id;
        /**
         * 具体任务线程
         */
        private Runnable runnable;
        /**
         * cron表达式
         */
        private String cron;
    }

}
