package com.onsiteservice.dao.common.page;

import com.github.pagehelper.Page;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;
import lombok.Getter;
import lombok.Setter;
import lombok.ToString;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * @author 潘维吉
 * @date 2018-05-19
 * 分页自定义一个PageInfoVO 替代PageHelper的PageInfo.java 他定义得有点重量级，属性有点多
 * 只有紧跟在 PageHelper.startPage(page, size) 方法后的第一个 Mybatis 的查询（Select）方法会被分页
 */
@SuppressWarnings("unchecked")
@ApiModel(description = "分页接口响应数据")
@Getter
@Setter
@ToString
public class PageInfoVO<T> implements Serializable {
    /**
     * Serializable是实现java将内存中的类存储至硬盘中而使用的
     * 如果没有指定serialVersionUID，系统会自动生成一个 对类进行版本控制的
     * serialVersionUID作用： 序列化时为了保持版本的兼容性，即在版本升级时反序列化仍保持对象的唯一性
     */
    private static final long serialVersionUID = 1L;

    /**
     * 当前页
     */
    @ApiModelProperty(value = "当前页", example = "1")
    private Integer page;

    /**
     * 当前页的数量
     */
    @ApiModelProperty("当前页的数量")
    private Integer size;

    /**
     * 总页数
     */
    @ApiModelProperty("总页数")
    private Integer pages;

    /**
     * 总记录数
     */
    @ApiModelProperty("总记录数")
    private Long total;

    /**
     * 是否有下一页
     */
    @ApiModelProperty("是否有下一页")
    private Boolean hasNextPage;

    /**
     * 数据集合
     */
    @ApiModelProperty("数据集合")
    private List<T> list;

    public PageInfoVO() {
        this.page = 1;
        this.size = 0;
        this.pages = 1;
        this.total = 0L;
        this.hasNextPage = false;
        this.list = new ArrayList<>();
    }

    public PageInfoVO(List<T> list) {
        if (list == null) {
            list = new ArrayList<>();
        }
        if (list instanceof Page) {
            Page page = (Page) list;
            this.page = page.getPageNum();
            this.size = page.size();
            this.pages = page.getPages();
            this.total = page.getTotal();
            this.hasNextPage = page.getPageNum() < page.getPages();
            this.list = page;
        } else if (list instanceof Collection) {
            this.page = 1;
            this.size = list.size();
            this.pages = 1;
            this.total = (long) list.size();
            this.hasNextPage = false;
            this.list = list;
        }
    }

}
