package com.onsiteservice.core.security.jwt;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.onsiteservice.core.exception.ServiceException;
import io.jsonwebtoken.Claims;
import io.jsonwebtoken.JwtBuilder;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.SignatureAlgorithm;
import io.jsonwebtoken.security.Keys;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.binary.Base64;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import javax.crypto.SecretKey;
import java.util.Date;
import java.util.Map;

/**
 * @author 潘维吉
 * @date 2018-05-20
 * JWT管理工具 token创建 解析 验证是否过期等
 */
@Lazy(false)
@ConfigurationProperties(prefix = "project.jwt")
@Component
@Getter
@Slf4j
public class JwtManager {

    /** http认证头常量 */
    public static final String AUTHORIZATION_HEADER = "Authorization";
    /** Bearer关键字常量 */
    public static final String BEARER = "Bearer ";

    /** Token中subject数据规定的key值 */
    public static final String USER_ID = "userId";
    // 以下为扩展存储token数据key 方便从token直接获取常用信息
    public static final String USER_NAME = "userName";
    public static final String PLATFORM_TYPE = "platformType";
    public static final String PLATFORM_ID = "platformId";
    public static final String COMPANY_ID = "companyId";

    /**
     * 密钥 签证信息 保存服务器端用来进行jwt的签发和jwt的验证，服务端的私钥，在任何场景都不应该流露出去
     * 签名的目的主要是为了验证我是“我” 于篡改者不知道密钥是什么，也无法生成新的 signature 部分，服务端也就无法通过，
     * 在 jwt 中，消息体是透明的，使用签名可以保证消息体不被篡改
     */
    public static String secret;
    /**
     * JWT的签发者
     */
    public static String issuer;
    /**
     * token过期时间 秒 -1不过期
     */
    public static long expiresTime;

    /**
     * 默认使用yaml文件统一配置的过期时间
     *
     * @param subject
     * @return
     */
    public static String createToken(Map<String, Object> subject) {
        return genToken(subject, expiresTime);
    }

    /**
     * 自定义过期时间
     *
     * @param subject
     * @param expireTime 过期时间 秒
     * @return
     */
    public static String createToken(Map<String, Object> subject, Long expireTime) {
        return genToken(subject, expireTime);
    }

    /**
     * 生成JWT token令牌
     *
     * @param subject    主题 用户id必填 key如 {"userId":1}
     * @param expireTime (可选) 过期时间
     * @return token信息
     */
    private static String genToken(Map<String, Object> subject, Long expireTime) {
        if (!subject.containsKey(USER_ID)) {
            throw new ServiceException("数据定义错误, 请使用 '" + USER_ID + "' 关键字作为用户Id的key值");
        }
        long nowMillis = System.currentTimeMillis();
        Date nowDate = new Date(nowMillis);

        //生成签名密钥 我们将用我们的ApiKey secret来签名我们的JWT
        SecretKey secretKey = Keys.hmacShaKeyFor(secret.getBytes());

        JSONObject jsonObject = new JSONObject();
        jsonObject.putAll(subject);

        //添加构成JWT的参数 Header Payload Sign信息
        JwtBuilder builder = Jwts.builder()
                .setHeaderParam("typ", "JWT")
                .setIssuedAt(nowDate) //创建时间
                .setSubject(jsonObject.toJSONString()) //主题,用户信息 JWT所面向的用户，是否使用是可选的
                .setIssuer(issuer) //该JWT的签发者，是否使用是可选的
                .signWith(secretKey, SignatureAlgorithm.HS256); //密钥签名


        //添加Token过期时间
        if (expireTime >= 0) {
            //过期时间
            long expireMillis = nowMillis + expireTime * 1000;
            //超期时间
            Date expireDate = new Date(expireMillis);
            //系统时间之前的token都是不可以被承认的
            builder.setExpiration(expireDate).setNotBefore(nowDate);
        }

        //生成JWT
        return builder.compact();
    }

    /**
     * 解析JWT 从令牌中获取数据声明
     *
     * @param token
     */
    public static Claims parseToken(String token) {
        Claims claims = null;
        try {
            claims = Jwts.parser()
                    .setSigningKey(secret.getBytes())
                    .parseClaimsJws(token).getBody();
            return claims;
        } catch (Exception e) {
            log.error("token解析错误: {}, token值: {}", e.getMessage(), token);
            if (claims != null) {
                log.error("claims数据: {}, 当前时间: {}", JSON.toJSONString(claims), new Date());
            } else {
                log.error("claims数据为null");
            }
            // 异常返回null 调用的地方根据null判断是否能够解析
            return null;
        }
    }

    /**
     * 判断令牌是否过期
     *
     * @param token 令牌
     * @return 是否过期
     */
    public static Boolean isExpired(String token) {
        try {
            Claims claims = parseToken(token);
            Date expiration = claims.getExpiration();
            return expiration.before(new Date());
        } catch (NullPointerException e) {
            return true;
        } catch (Exception e) {
            return false;
        }
    }

    /**
     * 获取token的过期时间
     *
     * @param token token信息
     * @return 过期时间
     */
    public static Date getExpirationTime(String token) {
        return parseToken(token).getExpiration();
    }

    /**
     * 设置过期
     *
     * @param token token信息
     */
    public static void setExpire(String token) {
        // 配合redis缓存等  删除缓存里当前token
    }

    @Value("${secret:cHJvamVjdC1uYW1lLWJhc2U2NC1zZWNyZXQ=}")
    public void setSecret(String secret) {
        JwtManager.secret = secret;
    }

    @Value("${issuer:issuer}")
    public void setIssuer(String issuer) {
        JwtManager.issuer = issuer;
    }

    @Value("${expires-time:-1}")
    public void setExpiresTime(long expiresTime) {
        JwtManager.expiresTime = expiresTime;
    }

    public static void main(String[] args) {
        // 生成base64的secret密钥  填写到yaml文件配置
        System.out.println(Base64.encodeBase64String("onsite-service-admin-base64-secret".getBytes()));
        System.out.println(Base64.encodeBase64String("onsite-service-app-base64-secret".getBytes()));
    }

}
