package com.onsiteservice.core.config;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.alibaba.fastjson.serializer.ValueFilter;
import com.alibaba.fastjson.support.config.FastJsonConfig;
import com.alibaba.fastjson.support.spring.FastJsonHttpMessageConverter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.http.HttpMessageConverters;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.http.MediaType;

import java.nio.charset.Charset;
import java.util.Arrays;

/**
 * @author 潘维吉
 * @date 2018-12-10 8:43
 * FastJson 统一配置
 * http返回结果统一处理
 */
@ConditionalOnProperty(prefix = "project.fast-json", name = {"enabled"}, matchIfMissing = true)
@Configuration
public class JsonConfig {

    /**
     * 使用阿里 FastJson 作为JSON 消息转换 提高JSON序列化速度 http返回结果处理等
     */
    @Bean
    public HttpMessageConverters fastJsonHttpMessageConverters() {
        //创建FastJson信息转换对象
        FastJsonHttpMessageConverter converter = new FastJsonHttpMessageConverter();
        //创建FastJsonConfig对象并设定序列化规则
        FastJsonConfig fastJsonConfig = new FastJsonConfig();
        //fastJsonConfig.setFeatures(Feature.OrderedField);
        fastJsonConfig.setSerializerFeatures(
                //保留null的字段 是否输出值为null的字段,默认为false
                SerializerFeature.WriteMapNullValue,
                //Date使用ISO8601格式输出 不使用时间戳, 但时间戳兼容性最好
                //SerializerFeature.UseISO8601DateFormat,
                //设置枚举类WriteEnumUsingToString 或 WriteEnumUsingName
                SerializerFeature.WriteEnumUsingToString,
                //List字段如果为null,输出为[],而非null 需要配合WriteMapNullValue使用
                SerializerFeature.WriteNullListAsEmpty,
                //禁止循环引用的方案 导致json返回出现 $ref
                SerializerFeature.DisableCircularReferenceDetect
        );

        // fastJson值统一自定义过滤转换处理
        fastJsonConfig.setSerializeFilters((ValueFilter) (object, key, value) -> {
            if (value instanceof Enum) {
                value = parseObject(value);
            }
            if (value instanceof String) {
                value = parseArray(value);
            }
            return value;
        });

        //SerializerFeature.MapSortField // FastJson新版本Map的序列化没有做排序再输出，原因是通过TreeMap排序很影响性能
        //SerializerFeature.WriteNullStringAsEmpty // String null -> ""
        //SerializerFeature.WriteNullNumberAsZero // Number null -> 0
        //SerializerFeature.WriteNullListAsEmpty // List字段如果为null,输出为[],而非null 需要配合WriteMapNullValue使用
        //SerializerFeature.WriteNullBooleanAsFalse // Boolean字段如果为null,输出为false,而非null
        //SerializerFeature.SortField // 按字段名称排序后输出 默认为false
        //SerializerFeature.DisableCircularReferenceDetect  // 禁止循环引用的方案 导致json返回出现 $ref
        //SerializerFeature.WriteSlashAsSpecial // 对斜杠’/’进行转义
        //SerializerFeature.PrettyFormat // 结果是否格式化,默认为false

        //规则赋予转换对象
        converter.setFastJsonConfig(fastJsonConfig);
        converter.setDefaultCharset(Charset.forName("UTF-8"));
        //默认是MediaType.ALL
        //converter.setSupportedMediaTypes(Arrays.asList(MediaType.APPLICATION_JSON));
        converter.setSupportedMediaTypes(Arrays.asList(MediaType.APPLICATION_JSON_UTF8));
        return new HttpMessageConverters(converter);
    }

    /**
     * json字符串转json对象
     *
     * @param value
     * @return json对象
     */
    private Object parseObject(Object value) {
        String v = value.toString();
        if (v.startsWith("{") && v.endsWith("}")) {
            try {
                value = JSON.parseObject(v);
            } catch (Exception e) {
                return value;
            }
        }
        return value;
    }

    /**
     * json数组字符串转json数组对象
     *
     * @param value
     * @return 数组
     */
    private Object parseArray(Object value) {
        String v = value.toString();
        if (v.startsWith("[") && v.endsWith("]")) {
            try {
                value = JSON.parseArray(v);
            } catch (Exception e) {
                return value;
            }
        }
        return value;
    }

    /**
     * 含分号,字符串转数组对象
     *
     * @param value
     * @return 数组
     */
    private Object toArray(Object value) {
        String v = value.toString();
        if (v.contains(",")) {
            value = v.split(",");
        }
        return value;
    }

}
