package com.onsiteservice.util.tree;

import com.onsiteservice.util.ReflectUtils;
import lombok.Getter;
import lombok.Setter;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;

import java.util.List;

import static com.google.common.base.Joiner.on;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Strings.emptyToNull;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Lists.reverse;
import static java.lang.String.format;


/**
 * @author 潘维吉
 * @date 2020/4/15 11:07
 * @email 406798106@qq.com
 * @description 动态生成每个节点的treePath路径
 */
public class TreePathUtils {

    private static final String DELIMITER = ","; // treePath全路径分隔符
    private static final String KEY = "id"; // 唯一标识
    private static final String PARENT_KEY = "parentId"; // 父类唯一标识
    private static final String TREE_PATH = "treePath"; // tree结构全路径

    /**
     * 根据平铺树计算路径
     */
    public static <T> void genPath(List<T> nodes) {
        for (T node : nodes) {
            List<String> path = newArrayList();
            recurNode(nodes, node, node, path, KEY, PARENT_KEY, TREE_PATH, DELIMITER, null);
        }
    }

    public static <T> void genPath(List<T> nodes, String delimiter) {
        for (T node : nodes) {
            List<String> path = newArrayList();
            recurNode(nodes, node, node, path, KEY, PARENT_KEY, TREE_PATH, delimiter, null);
        }
    }

    public static <T> void genPath(List<T> nodes, String key, String parentKey) {
        for (T node : nodes) {
            List<String> path = newArrayList();
            recurNode(nodes, node, node, path, key, parentKey, TREE_PATH, DELIMITER, null);
        }
    }

    public static <T> void genPath(List<T> nodes, String key, String parentKey, String treePath, String delimiter) {
        for (T node : nodes) {
            List<String> path = newArrayList();
            recurNode(nodes, node, node, path, key, parentKey, treePath, delimiter, null);
        }
    }

    public static <T> void genPath(List<T> nodes, String key, String parentKey, String treePath, String delimiter, String treePathValue) {
        for (T node : nodes) {
            List<String> path = newArrayList();
            recurNode(nodes, node, node, path, key, parentKey, treePath, delimiter, treePathValue);
        }
    }

    /**
     * 递归查找父节点
     *
     * @param nodes       所有节点
     * @param targetNode  目标节点
     * @param currentNode 当前节点
     * @param path        路径
     */
    private static <T> void recurNode(List<T> nodes, T targetNode, T currentNode, List<String> path,
                                      String key, String parentKey, String treePath, String delimiter, String treePathValue) {
/*        if (currentNode.getId().equals(currentNode.getParentId())) {
            throw new RuntimeException(format("非法的树结构,node:{0}",
                    currentNode.getId()));
        }*/
        path.add(checkNotNull(emptyToNull(ReflectUtils.invokeGetter(
                currentNode, StringUtils.isNotBlank(treePathValue) ? treePathValue : key).toString()),
                format("节点编码为空")));
        // 终止条件,这里约定null,""表示根节点
        if (ObjectUtils.isEmpty(ReflectUtils.invokeGetter(currentNode, parentKey))) {
            ReflectUtils.invokeSetter(targetNode, treePath, on(delimiter).join(reverse(path)));
            return;
        }
        // 节点编号必须唯一,每次只能找到一个父节点
        for (T node : nodes) {
            if (ReflectUtils.invokeGetter(currentNode, parentKey)
                    .equals(ReflectUtils.invokeGetter(node, key))) {
                recurNode(nodes, targetNode, node, path, key, parentKey, treePath, delimiter, treePathValue);
                return;
            }
        }
        // 既不是根节点又无法找到父节点
/*        throw new RuntimeException(format("非法的树结构,node:{0}",
                currentNode.getId()));*/
    }

    /**
     * 根据需求自定义组装Node VO对象
     */
    @Getter
    @Setter
    public static class Node {
        private Long id;
        private Long parentId;
        private String name;
        private String treePath;

        public Node(Long id, String name, Long parentId) {
            this.id = id;
            this.parentId = parentId;
            this.name = name;
        }
    }

    /**
     * @param args
     */
    public static void main(String[] args) {
        List<Node> tree = newArrayList();

        // 第一颗课树
        tree.add(new Node(1L, "1", null));
        tree.add(new Node(2L, "1-1", 1L));
        tree.add(new Node(3L, "1-1-1", 2L));
        tree.add(new Node(4L, "1-1-2", 2L));
        // 第二
        tree.add(new Node(5L, "2", null));
        tree.add(new Node(6L, "2-1", 5L));
        tree.add(new Node(7L, "2-1-1", 6L));
        genPath(tree);
        // genPath(tree,"id","parentId","path");
        for (Node node : tree) {
            System.out.println(node.getId() + " -> " + node.getTreePath());
        }

    }

}


