package com.onsiteservice.util.codec;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.codec.digest.DigestUtils;

import javax.crypto.Cipher;
import javax.crypto.KeyGenerator;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.security.Key;
import java.security.SecureRandom;


/**
 * @author 潘维吉
 * @date 2018-05-20
 * 编码加解密相关工具类
 */
public class CodecUtils {
    private static final String DEFAULT_URL_ENCODE = "UTF-8";

    /**
     * base64进制编码
     *
     * @param content
     * @return
     */
    public static String encodeBase64(String content) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(content), "不能为空");
        byte[] bytes = content.getBytes();
        return Base64.encodeBase64String(bytes);
    }

    /**
     * base64进制解码
     *
     * @param content
     * @return
     */
    public static String decodeBase64(String content) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(content), "消息摘要不能为空");
        return new String(Base64.decodeBase64(content));
    }

    /**
     * MD5加密
     *
     * @param content
     * @return
     */
    public static String md5(String content) {
        return DigestUtils.md5Hex(content);
    }

    /**
     * SHA加密
     *
     * @param content
     * @return
     */
    public static String sha(String content) {
        return DigestUtils.sha1Hex(content);
    }

    /**
     * AES加密
     *
     * @param key     加密Key
     * @param content 加密内容
     * @return
     */
    public static String encryptAES(String key, String content) {
        Key secretKey = getAesKey(key);
        try {
            Cipher cipher = Cipher.getInstance("AES");
            cipher.init(Cipher.ENCRYPT_MODE, secretKey);
            byte[] p = content.getBytes("UTF-8");
            byte[] result = cipher.doFinal(p);
            String encoded = Base64.encodeBase64String(result);
            return encoded;
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * AES解密
     *
     * @param key     解密Key
     * @param content 解密内容
     * @return
     */
    public static String decryptAES(String key, String content) {
        Key secretKey = getAesKey(key);
        try {
            Cipher cipher = Cipher.getInstance("AES");
            cipher.init(Cipher.DECRYPT_MODE, secretKey);
            byte[] c = Base64.decodeBase64(content);
            byte[] result = cipher.doFinal(c);
            String plainText = new String(result, "UTF-8");
            return plainText;
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * AES生成Key
     *
     * @param key 解密Key
     * @return
     */
    public static Key getAesKey(String key) {
        if (key == null) {
            key = System.getenv("AES_SYS_KEY");
        }
        if (key == null) {
            key = System.getProperty("AES_SYS_KEY");
        }
        if (key == null || key.trim().length() == 0) {
            key = "abcd1234!@#$";// 默认种子
        }
        try {
            SecureRandom secureRandom = SecureRandom.getInstance("SHA1PRNG");
            secureRandom.setSeed(key.getBytes());
            KeyGenerator generator = KeyGenerator.getInstance("AES");
            generator.init(secureRandom); // 密钥长度是在生成密钥时指定的
            return generator.generateKey();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Hex编码.
     */
    public static String encodeHex(byte[] input) {
        return new String(Hex.encodeHex(input));
    }

    /**
     * Hex解码.
     */
    public static byte[] decodeHex(String input) {
        try {
            return Hex.decodeHex(input.toCharArray());
        } catch (DecoderException e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * URL 编码
     */
    public static String encodeUrl(String part) {
        try {
            return URLEncoder.encode(part, DEFAULT_URL_ENCODE);
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
        }
        return "";
    }

    /**
     * URL 解码
     */
    public static String decodeUrl(String part) {
        try {
            return URLDecoder.decode(part, DEFAULT_URL_ENCODE);
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
        }
        return "";
    }

    /**
     * base64转16进制
     * @param message
     */
    public static String base64ToHex(String message){
        final java.util.Base64.Decoder decoder = java.util.Base64.getDecoder();
        byte[] decode = decoder.decode(message);
        return byte2Hex(decode);
    }

    /**
     * 2进制转16进制
     * @param bytes
     * @return
     */
    public static String byte2Hex(byte[] bytes) {
        StringBuffer stringBuffer = new StringBuffer();
        String temp = null;
        for (int i = 0; i < bytes.length; i++) {
            temp = Integer.toHexString(bytes[i] & 0xFF);
            if (temp.length() == 1) {
                // 1得到一位的进行补0操作
                stringBuffer.append("0");
            }
            stringBuffer.append(temp);
        }
        return stringBuffer.toString();
    }

    /**
     * 16进制字符串转byte[]数组
     * @param s
     * @return
     */
    public static byte[] hexStringToByteArray(String s) {
        int len = s.length();
        byte[] data = new byte[len / 2];
        for (int i = 0; i < len; i += 2) {
            data[i / 2] = (byte) ((Character.digit(s.charAt(i), 16) << 4)
                    + Character.digit(s.charAt(i + 1), 16));
        }
        return data;
    }

    public static void main(String[] args) {
        String a1 = encodeBase64("66666");
        System.out.println(a1);
        String a2 = decodeBase64(a1);
        System.out.println(a2);
        System.out.println(md5("123456"));
        System.out.println(sha("测试"));
        String aes = encryptAES("panweiji", "这是AES加密的信息");
        System.out.println(aes);
        System.out.println(decryptAES("panweiji", aes));

    }
}
