package com.onsiteservice.common.service;

import com.alibaba.fastjson.JSON;
import com.google.common.collect.ImmutableMap;
import com.onsiteservice.common.controller.convert.BaseConvert;
import com.onsiteservice.common.controller.vo.AreaTreeVO;
import com.onsiteservice.common.redis.RedisUtils;
import com.onsiteservice.constant.constant.RedisKeyConstants;
import com.onsiteservice.constant.enums.FileTypeEnum;
import com.onsiteservice.core.exception.ServiceException;
import com.onsiteservice.core.result.Result;
import com.onsiteservice.dao.mapper.base.BaseAreaMapper;
import com.onsiteservice.dao.mapper.sys.SysParamMapper;
import com.onsiteservice.entity.base.BaseArea;
import com.onsiteservice.entity.sys.SysParam;
import com.onsiteservice.util.ArrayUtils;
import com.onsiteservice.util.UUIDUtils;
import com.onsiteservice.util.aliyun.OssUtils;
import com.onsiteservice.util.tree.TreeUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import java.io.IOException;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import static com.onsiteservice.core.result.ResultGenerator.fail;
import static com.onsiteservice.core.result.ResultGenerator.success;


/**
 * @author 潘维吉
 * @date 2020/7/1 15:08
 * @email 406798106@qq.com
 * @description 通用基础业务服务
 */
@Service
@Transactional(rollbackFor = Exception.class)
public class CommonService {

    /**
     * 自定义系统工程名称 用于区分不同项目模块
     */
    @Value("${spring.application.name:project}")
    private String projectName;

    @Resource
    private OssUtils ossUtils;
    @Resource
    private RedisUtils redisUtils;
    @Resource
    private BaseAreaMapper baseAreaMapper;
    @Resource
    private BaseConvert baseConvert;
    @Resource
    private SysParamMapper sysParamMapper;

    /**
     * 统一上传oss服务
     */
    public Result uploadOss(MultipartFile[] files, FileTypeEnum fileType, Long userId) {
        List urls = new ArrayList<>();
        Arrays.asList(files).forEach(file -> {
            try {
                String originalFilename = file.getOriginalFilename();
                //获取文件后缀类型
                String fileSuffix = originalFilename.substring(originalFilename.lastIndexOf("."));
                String fileTypeName = fileType.name().toLowerCase();
                String filePath = "";
                if (fileTypeName.equalsIgnoreCase(FileTypeEnum.SYSTEM.toString())) { // 系统
                    filePath = fileTypeName;
                } else if (fileTypeName.equalsIgnoreCase(FileTypeEnum.RICH_TEXT.toString())) { // 富文本
                    filePath = projectName + "/" + fileTypeName;
                } else {
                    filePath = projectName + "/" + fileTypeName + "/" + userId;
                }
                filePath = filePath + "/" + UUIDUtils.getFlowID() + fileSuffix;

                String httpUrl = ossUtils.uploadFile(filePath, file.getBytes());
                urls.add(httpUrl);
            } catch (IOException e) {
                throw new ServiceException("上传文件异常: " + e.getMessage());
            }
        });

        if (!ArrayUtils.isEmpty(urls)) {
            return success(ImmutableMap.of("urls", urls), "上传文件成功");
        } else {
            return fail("上传文件失败");
        }
    }

    /**
     * 统一删除oss服务
     */
    public Result deleteOss(String path) {
        if (StringUtils.isNotBlank(path)) {
            try {
                ossUtils.deleteFile(path);
                return success(String.format("删除文件 %s 成功", path));
            } catch (Exception e) {
                return success(String.format("删除文件 %s 失败", path));
            }
        }
        return success(String.format("文件 %s 不存在", path));
    }

    /**
     * 获取所有区域树
     */
    public List<AreaTreeVO> getAllTree() {
        if (redisUtils.exists(RedisKeyConstants.BASE_AREA_TREE)) {
            return JSON.parseObject(redisUtils.get(RedisKeyConstants.BASE_AREA_TREE).toString(), List.class);
        } else {
            List<AreaTreeVO> areaTreeVOList = baseConvert.toAreaTreeVO(
                    baseAreaMapper.selectAll().stream()
                            .filter(item -> !"street".equals(item.getLevel()))
                            .sorted(Comparator.comparing(BaseArea::getShowOrder).reversed())
                            .collect(Collectors.toList()));
            List list = TreeUtils.getTree(areaTreeVOList,
                    AreaTreeVO::getCode, AreaTreeVO::getParentCode, AreaTreeVO::setChildren);
            redisUtils.set(RedisKeyConstants.BASE_AREA_TREE, list, 30L, TimeUnit.DAYS);
            return list;
        }
    }

    public String getSysParamValue(String code) {
        SysParam sysParam = new SysParam();
        sysParam.setCode(code);
        SysParam param = sysParamMapper.selectOne(sysParam);
        return Objects.isNull(param) ? "" : param.getValue();
    }

    public Integer getSysParamIntValue(String code) {
        String str = this.getSysParamValue(code);
        if (StringUtils.isEmpty(str)) {
            return 0;
        }
        return Integer.parseInt(str);
    }

}
