package com.onsiteservice.common.annotation.validation;

import com.onsiteservice.core.exception.ServiceException;
import com.onsiteservice.dao.common.AbstractMapper;
import com.onsiteservice.util.ReflectUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;
import org.aspectj.lang.annotation.Pointcut;
import org.springframework.stereotype.Component;
import tk.mybatis.mapper.entity.Condition;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.List;
import java.util.Objects;

@Slf4j
@Aspect
@Component
public class UniqueCheckAspect {

    @Pointcut("@annotation(com.onsiteservice.common.annotation.validation.Unique)")
    public void unique() {
    }

    @Before("unique()")
    public void before(JoinPoint joinPoint) {
        var obj = joinPoint.getArgs()[0]; // 获取参数对象
        AbstractMapper mapper = (AbstractMapper) joinPoint.getThis(); // 获取service
        var uniqueCheck = getUniqueCheck(joinPoint); // 获取注解对象
        String[] field = uniqueCheck.field();
        var conditionFields = uniqueCheck.conditionFields();
        var message = uniqueCheck.message();

        var modelClass = obj.getClass();

        while (!isModel(modelClass)) {
            modelClass = modelClass.getSuperclass();
        }
        for(int i=0;i<field.length; i++) {
            String f = field[i];
            Condition condition = new Condition(modelClass);
            Condition.Criteria criteria = condition.createCriteria();
            try {
                criteria.andEqualTo(f, BeanUtils.getNestedProperty(obj, f));
                Object id = ReflectUtils.invokeGetter(obj, "id");
                if (ObjectUtils.isNotEmpty(id)) {
                    // 有id,在除了该条数据外的所有数据中查询
                    criteria.andNotEqualTo("id", id);
                }
                for (String conditionField : conditionFields) {
                    criteria.andEqualTo(conditionField, ReflectUtils.invokeGetter(obj, conditionField));
                }
            } catch (IllegalAccessException | InvocationTargetException | NoSuchMethodException e) {
                e.printStackTrace();
            }
            if (mapper.selectCountByCondition(condition) != 0) {
                throw new ServiceException(message[i]);
            }
        }

    }

    /**
     * 判断某类型是不是Model, 根据有没有id字段判断
     */
    private boolean isModel(Class<?> clazz) {
        return List.of(clazz.getDeclaredFields()).stream().anyMatch(f -> "id".equals(f.getName()));
    }

    /**
     * 获取注解对象
     *
     * @param joinPoint 对象
     */
    private Unique getUniqueCheck(final JoinPoint joinPoint) {
        Method[] methods = joinPoint.getTarget().getClass().getDeclaredMethods();
        String name = joinPoint.getSignature().getName();
        if (!StringUtils.isEmpty(name)) {
            for (Method method : methods) {
                Unique annotation = method.getAnnotation(Unique.class);
                if (!Objects.isNull(annotation) && name.equals(method.getName())) {
                    return annotation;
                }
            }
        }
        return null;
    }
}
