package com.onsiteservice.admin.service.sys;

import com.onsiteservice.admin.controller.sys.dto.PwdChangeDTO;
import com.onsiteservice.admin.controller.sys.dto.SysUserDetailDTO;
import com.onsiteservice.admin.controller.sys.dto.SysUserPageDTO;
import com.onsiteservice.admin.controller.sys.vo.SysUserDetailVO;
import com.onsiteservice.admin.mapper.sys.SysUserBizMapper;
import com.onsiteservice.common.service.CommonService;
import com.onsiteservice.constant.constant.SysParamConstants;
import com.onsiteservice.core.exception.ServiceException;
import com.onsiteservice.dao.mapper.sys.SysUserMapper;
import com.onsiteservice.dao.util.EntityUtils;
import com.onsiteservice.entity.sys.SysUser;
import com.onsiteservice.util.CommonUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.security.crypto.bcrypt.BCryptPasswordEncoder;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.List;

@Service
@Transactional(rollbackFor = Exception.class)
public class SysUserService {

    @Value("${project.default-pwd}")
    private String defaultPassword;
    @Value("${aliyun.oss.default-avatar}")
    private String defaultAvatar;

    @Resource
    private SysUserBizMapper sysUserBizMapper;
    @Resource
    private SysUserMapper sysUserMapper;
    @Resource
    private PasswordEncoder passwordEncoder;

    @Resource
    private SysRoleService sysRoleService;

    @Resource
    private SysUserRoleService sysUserRoleService;

    @Resource
    private BCryptPasswordEncoder bCryptPasswordEncoder;
    @Resource
    private CommonService commonService;

    public List<SysUser> getPage(SysUserPageDTO dto) {
        return sysUserBizMapper.getPage(dto);
    }

    public SysUser getById(Long id) {
        SysUser user = sysUserMapper.selectByPrimaryKey(id);

        if (ObjectUtils.isEmpty(user)) {
            throw new ServiceException(401, "找不到该用户");
        }
        if (ObjectUtils.isEmpty(user.getAvatarUrl())) {
            user.setAvatarUrl(defaultAvatar);
        }
        return user;
    }


    public SysUserDetailVO getUserDetail(Integer id) {
        SysUserDetailVO result = new SysUserDetailVO();
        if (CommonUtils.isZero(id)) {
            result.setSysUserDetail(new SysUser());
        } else {
            result.setSysUserDetail(sysUserMapper.selectByPrimaryKey(id));
            result.setSysUserCheckedRoleList(sysUserRoleService.getUserRoleByUserId(id));
        }
        result.setSysUserRoleList(sysRoleService.getAllAvailableRole());
        return result;

    }

    public long saveDetail(SysUserDetailDTO dto, Long userId) {
        //待保存用户信息
        SysUser newUserInfo = dto.getSysUserDetail();
        // 校验用户名是否重复
        if (sysUserBizMapper.getRepeat(newUserInfo.getId() == null ? 0 : newUserInfo.getId(), newUserInfo.getUserName()) > 0) {
            throw new ServiceException("该用户名已存在，请修改！");
        }
        if(newUserInfo.getAvatarUrl()==null){
            newUserInfo.setAvatarUrl(commonService.getSysParamValue(SysParamConstants.AVATAR_URL));
        }
        // 勾选角色
        List<Integer> sysUserCheckedRoleList = dto.getSysUserCheckedRoleList();
        // 根据数据有无id判断是新建还是修改
        if (CommonUtils.isZero(newUserInfo.getId())) {
            this.setSysUserDefaultInfo(newUserInfo);
            sysUserMapper.insertSelective(newUserInfo);
        } else {
            // 更新，同时清空redis
            EntityUtils.update(newUserInfo,userId);
            this.saveCache(newUserInfo);
        }
        // 先删除
        sysUserRoleService.deleteByUserId(newUserInfo.getId());
        if (null != sysUserCheckedRoleList && sysUserCheckedRoleList.size() > 0) {
            // 再增加
            sysUserRoleService.insertUserRole(newUserInfo.getId(), sysUserCheckedRoleList);
        }
        return newUserInfo.getId();
    }

    /**
     * 设置SysUser默认信息
     *
     * @param sysUser
     */
    public void setSysUserDefaultInfo(SysUser sysUser) {
        // 默认密码
        sysUser.setPassword(bCryptPasswordEncoder.encode(defaultPassword));
        // 默认启用状态
        sysUser.setIsEnabled(true);
    }

    public void saveCache(SysUser user) {
        sysUserMapper.updateByPrimaryKeySelective(user);
    }

    public void resetPwd(Integer id) {
        SysUser sysUser = sysUserMapper.selectByPrimaryKey(id);
        sysUser.setPassword(bCryptPasswordEncoder.encode(defaultPassword));
        this.saveCache(sysUser);
    }

    public void editPwd(PwdChangeDTO dto, Long userId) {
        if (StringUtils.isBlank(dto.getNewPwd()) || StringUtils.isBlank(dto.getOldPwd())) {
            throw new ServiceException("新登录密码为空,请确认！");
        }
        SysUser user = sysUserMapper.selectByPrimaryKey(userId);
        if (bCryptPasswordEncoder.matches(dto.getOldPwd(), user.getPassword())) {
            user.setPassword(bCryptPasswordEncoder.encode(dto.getNewPwd()));
            // 保存，并清空redis
            saveCache(user);
        } else {
            throw new ServiceException("当前登录密码错误,请确认！");
        }
    }

    public Long changeStatus(Long id) {
        SysUser sysUser = sysUserMapper.selectByPrimaryKey(id);
        sysUser.setIsEnabled(!sysUser.getIsEnabled());
        // 更新同时清空redis
        saveCache(sysUser);
        return sysUser.getId();
    }
}
