package com.onsiteservice.admin.service.service;

import com.github.pagehelper.PageHelper;
import com.onsiteservice.admin.controller.service.dto.PageServiceWorkerDTO;
import com.onsiteservice.admin.controller.service.dto.SaveServiceWorkerDTO;
import com.onsiteservice.admin.controller.service.dto.UpdateServiceWorkerDTO;
import com.onsiteservice.admin.controller.service.vo.ServiceWorkerKindVO;
import com.onsiteservice.admin.controller.service.vo.ServiceWorkerVO;
import com.onsiteservice.admin.mapper.service.ServiceWorkerBizMapper;
import com.onsiteservice.constant.enums.BizCodeEnum;
import com.onsiteservice.constant.enums.ServiceUserTypeEnum;
import com.onsiteservice.constant.enums.ServiceWorkerEnum;
import com.onsiteservice.core.exception.ServiceException;
import com.onsiteservice.core.result.Result;
import com.onsiteservice.core.result.ResultGenerator;
import com.onsiteservice.dao.common.AbstractMapper;
import com.onsiteservice.dao.common.page.PageInfoVO;
import com.onsiteservice.dao.mapper.service.ServiceWorkerMapper;
import com.onsiteservice.dao.mapper.user.UserMapper;
import com.onsiteservice.entity.service.ServiceWorker;
import com.onsiteservice.entity.user.User;
import com.onsiteservice.util.AttrCopyUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import tk.mybatis.mapper.entity.Condition;

import javax.annotation.Resource;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

@Service
@Slf4j
@Transactional(rollbackFor = Exception.class)
public class ServiceWorkerService extends AbstractMapper<ServiceWorker> {

    @Resource
    private ServiceWorkerMapper serviceWorkerMapper;

    @Resource
    private ServiceWorkerBizMapper serviceWorkerBizMapper;

    @Resource
    private UserMapper userMapper;

    /**
     * 分页查询列表
     */
    public Result<PageInfoVO<ServiceWorkerVO>> getPage(PageServiceWorkerDTO dto) {
        log.info("worker getPage dto: {}", dto);

        PageHelper.startPage(dto.getPage(), dto.getSize());
        List<ServiceWorker> serviceWorkerList = serviceWorkerBizMapper.selectServiceWorkerPage(dto);
        List<ServiceWorkerVO> serviceWorkerVOList = serviceWorkerList.parallelStream().map(e -> AttrCopyUtils.copy(e, new ServiceWorkerVO())).collect(Collectors.toList());

        return ResultGenerator.success(new PageInfoVO<>(serviceWorkerVOList));
    }

    public ServiceWorkerVO selectById(Long id) {
        log.info("worker selectById id: {}", id);

        ServiceWorker serviceWorker = this.selectByPrimaryKey(id);
        if (Objects.isNull(serviceWorker)) {
            throw new ServiceException(BizCodeEnum.SERVICE_WORKER_NOT_EXIST);
        }

        return AttrCopyUtils.copy(serviceWorker, new ServiceWorkerVO());
    }

    public int save(SaveServiceWorkerDTO dto, Long userId) {
        log.info("worker save dto: {}, userId: {}", dto, userId);

        // checkAuth(userId);

        Condition c = new Condition(ServiceWorker.class);
        c.createCriteria().andEqualTo("phone", dto.getPhone());
        List<ServiceWorker> serviceWorkerList = this.selectByCondition(c);
        if (!CollectionUtils.isEmpty(serviceWorkerList)) {
            throw new ServiceException(BizCodeEnum.SERVICE_WORKER_HAS_EXIST);
        }

        return this.insertSelective(AttrCopyUtils.copy(dto, new ServiceWorker()));
    }

    public int update(UpdateServiceWorkerDTO dto, Long userId) {
        log.info("worker update dto: {}, userId: {}", dto, userId);

        // checkAuth(userId);

        return this.updateByPrimaryKeySelective(AttrCopyUtils.copy(dto, new ServiceWorker()));
    }

    public int delete(Long id, Long userId) {
        log.info("worker delete id: {}, userId: {}", id, userId);

        // checkAuth(userId);

        return this.deleteByPrimaryKey(id);
    }


    public List<ServiceWorkerKindVO> kind() {
        return Arrays.stream(ServiceWorkerEnum.values()).parallel()
                .map(e -> ServiceWorkerKindVO.builder().id(e.getId()).name(e.getName()).build())
                .collect(Collectors.toList());
    }


    private void checkAuth(Long userId) {
        User user = userMapper.selectByPrimaryKey(userId);
        if (Objects.isNull(user) || !user.getRoleType().equals(ServiceUserTypeEnum.ADMIN.getId())) {
            throw new ServiceException(BizCodeEnum.NO_AUTH);
        }
    }


}
