package com.onsiteservice.admin.service.service;

import com.github.pagehelper.PageHelper;
import com.onsiteservice.admin.controller.service.dto.PageServiceValuatorDTO;
import com.onsiteservice.admin.controller.service.dto.SaveServiceValuatorDTO;
import com.onsiteservice.admin.controller.service.dto.UpdateServiceValuatorDTO;
import com.onsiteservice.admin.controller.service.vo.ServiceValuatorVO;
import com.onsiteservice.admin.mapper.service.ServiceValuatorBizMapper;
import com.onsiteservice.constant.enums.BizCodeEnum;
import com.onsiteservice.constant.enums.ServiceUserTypeEnum;
import com.onsiteservice.core.exception.ServiceException;
import com.onsiteservice.core.result.Result;
import com.onsiteservice.core.result.ResultGenerator;
import com.onsiteservice.dao.common.AbstractMapper;
import com.onsiteservice.dao.common.page.PageInfoVO;
import com.onsiteservice.dao.mapper.user.UserMapper;
import com.onsiteservice.entity.service.ServiceValuator;
import com.onsiteservice.entity.service.ServiceWorker;
import com.onsiteservice.entity.user.User;
import com.onsiteservice.util.AttrCopyUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import tk.mybatis.mapper.entity.Condition;

import javax.annotation.Resource;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @author 潘维吉
 * @date 2022-07-11 16:07
 * @description ServiceValuatorService服务类
 */
@Service
@Slf4j
@Transactional(rollbackFor = Exception.class)
public class ServiceValuatorService extends AbstractMapper<ServiceValuator> {


    @Resource
    private ServiceValuatorBizMapper serviceValuatorBizMapper;

    @Resource
    private UserMapper userMapper;


    /**
     * 分页查询列表
     */
    public Result<PageInfoVO<ServiceValuatorVO>> getPage(PageServiceValuatorDTO dto) {
        PageHelper.startPage(dto.getPage(), dto.getSize());

        List<ServiceValuator> serviceValuatorList = serviceValuatorBizMapper.selectServiceValuatorPage(dto);
        List<ServiceValuatorVO> serviceValuatorVOList = serviceValuatorList.parallelStream().map(e -> AttrCopyUtils.copy(e, new ServiceValuatorVO())).collect(Collectors.toList());

        return ResultGenerator.success(new PageInfoVO<>(serviceValuatorVOList));
    }


    public ServiceValuatorVO selectById(Long id) {
        log.info("valuator selectById id: {}", id);

        ServiceValuator serviceValuator = this.selectByPrimaryKey(id);
        if (Objects.isNull(serviceValuator)) {
            throw new ServiceException(BizCodeEnum.SERVICE_VALUATOR_NOT_EXIST);
        }

        return AttrCopyUtils.copy(serviceValuator, new ServiceValuatorVO());
    }


    public int save(SaveServiceValuatorDTO dto, Long userId) {
        log.info("valuator save dto: {}, userId: {}", dto, userId);

        // checkAuth(userId);

        Condition c = new Condition(ServiceWorker.class);
        c.createCriteria().andEqualTo("phone", dto.getPhone());
        List<ServiceValuator> serviceValuatorList = this.selectByCondition(c);
        if (!CollectionUtils.isEmpty(serviceValuatorList)) {
            throw new ServiceException(BizCodeEnum.SERVICE_VALUATOR_HAS_EXIST);
        }

        return this.insertSelective(AttrCopyUtils.copy(dto, new ServiceValuator()));
    }


    public int update(UpdateServiceValuatorDTO dto, Long userId) {
        log.info("valuator update dto: {}, userId: {}", dto, userId);

        // checkAuth(userId);

        return this.updateByPrimaryKeySelective(AttrCopyUtils.copy(dto, new ServiceValuator()));
    }


    public int delete(Long id, Long userId) {
        log.info("valuator delete id: {}, userId: {}", id, userId);

        // checkAuth(userId);

        return this.deleteByPrimaryKey(id);
    }


    private void checkAuth(Long userId) {
        User user = userMapper.selectByPrimaryKey(userId);
        if (Objects.isNull(user) || !user.getRoleType().equals(ServiceUserTypeEnum.ADMIN.getId())) {
            throw new ServiceException(BizCodeEnum.NO_AUTH);
        }
    }


}
