package com.onsiteservice.miniapp.service.weixin;

import cn.binarywang.wx.miniapp.api.WxMaService;
import cn.binarywang.wx.miniapp.bean.*;
import com.google.common.collect.ImmutableMap;
import com.onsiteservice.common.service.dto.MiniQrCodeDTO;
import com.onsiteservice.constant.enums.EnvironmentEnum;
import com.onsiteservice.constant.enums.ServiceUserTypeEnum;
import com.onsiteservice.core.exception.ServiceException;
import com.onsiteservice.core.result.Result;
import com.onsiteservice.core.security.jwt.JwtManager;
import com.onsiteservice.entity.service.ServicePromote;
import com.onsiteservice.entity.user.User;
import com.onsiteservice.miniapp.controller.weixn.dto.LoginDTO;
import com.onsiteservice.miniapp.controller.weixn.dto.WxEncryptDTO;
import com.onsiteservice.miniapp.service.user.ServicePromoteService;
import com.onsiteservice.miniapp.service.user.UserService;
import io.micrometer.core.instrument.util.StringUtils;
import lombok.extern.slf4j.Slf4j;
import me.chanjar.weixin.common.error.WxErrorException;
import org.apache.commons.lang3.ObjectUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import static com.onsiteservice.core.result.ResultGenerator.fail;
import static com.onsiteservice.core.result.ResultGenerator.success;


/**
 * @author 潘维吉
 * @date 2021-09-14 08:33
 * 微信服务类
 */
@Slf4j
@Service
@Transactional(rollbackFor = Exception.class)
public class WeiXinService {


    @Resource
    private WxMaService wxMaService;
    @Resource
    private UserService userService;
    @Resource
    private ServicePromoteService promoteService;
    @Value("${spring.profiles.active:dev}")
    private String env;


    /**
     * 用户信息解密  登录或注册一体
     */
    public Result userLogin(LoginDTO loginDTO) {
        try {
            WxMaJscode2SessionResult sessionResult = null;
            try {
                // 获取openid、unionid、session_key等
                sessionResult = wxMaService.getUserService().getSessionInfo(loginDTO.getCode());
            } catch (WxErrorException e) {
                e.printStackTrace();
                throw new ServiceException("微信小程序登录失败:" + e.getMessage());
            }
            String unionId = sessionResult.getUnionid(); // unionId需要先在微信开放平台绑定平台
            String openId = sessionResult.getOpenid();
            String sessionKey = sessionResult.getSessionKey();

            // 用户信息校验 wx.login重复登录 SessionKey可能失效
            if (!wxMaService.getUserService().checkUserInfo(
                    sessionKey, loginDTO.getRawData(), loginDTO.getSignature())) {
                return fail("微信用户信息校验失败");
            }

/*            if (StringUtils.isBlank(unionId)) {
                return fail("unionId无法获取，登录失败");
            }*/
            // 解密用户信息
            WxMaUserInfo userInfo = wxMaService.getUserService().getUserInfo(
                    sessionKey, loginDTO.getEncryptedData(), loginDTO.getIv());
            String nickName = userInfo.getNickName();
            String avatarUrl = userInfo.getAvatarUrl();

            // 是否存在用户
            Optional<User> hasUser = Optional.ofNullable(userService.selectOneByProperty("openId", openId));
            if (hasUser.isPresent()) {
                User user = hasUser.get();
                user.setNickName(nickName);
                user.setAvatar(avatarUrl);
                userService.updateByPrimaryKeySelective(user);
                // 关联导入IM账号
                return toLogin(user, user.getId(), "登录成功");
            } else {
                // 新增用户
                User user = User.builder()
                        .userName(nickName)
                        .nickName(nickName)
                        .unionId(unionId)
                        .openId(openId)
                        .avatar(avatarUrl)
                        .sex(Integer.parseInt(userInfo.getGender()))
                        .roleType(ServiceUserTypeEnum.USER.getId())
                        .build();
                userService.insertSelective(user);
                if(ObjectUtils.isNotEmpty(loginDTO.getShareUserId())) {
                    promoteService.promoteCount(loginDTO.getShareUserId());
                }
                return toLogin(user, user.getId(), "注册成功");
            }

        } catch (Exception e) {
            log.error("登录失败:" + e.getMessage());
            throw new ServiceException("登录失败");
        }
    }

    /**
     * 登录  通用代码重构封装
     */
    private Result toLogin(User user, Long userId, String msg) {
        //生成一个token，保存用户登录状态
        String jwtToken = JwtManager.createToken(Map.of(JwtManager.USER_ID, userId));
        Map<String, Object> map = ImmutableMap.of("token", jwtToken,
                "userId", userId, "userInfo", user);
        return success(map, msg);
    }

    /**
     * 获取微信用户绑定手机号信息
     */
    public WxMaPhoneNumberInfo getPhone(WxEncryptDTO wxEncryptDTO, Long userId) {
        WxMaJscode2SessionResult json = null;
        try {
            json = wxMaService.getUserService().getSessionInfo(wxEncryptDTO.getCode());
        } catch (WxErrorException e) {
            e.printStackTrace();
            throw new ServiceException("微信小程序异常:" + e.getMessage());
        }
        wxEncryptDTO.setSessionKey(json.getSessionKey());
        // 解密手机号
        try {
            WxMaPhoneNumberInfo phoneInfo = wxMaService.getUserService().getPhoneNoInfo(
                    wxEncryptDTO.getSessionKey(), wxEncryptDTO.getEncryptedData(), wxEncryptDTO.getIv());
            updatePhone(userId, phoneInfo.getPhoneNumber());
            return phoneInfo;
        } catch (Exception e) {
            e.printStackTrace();
            log.error("获取微信手机号失败:" + e.getMessage());
            throw new ServiceException("获取微信手机号失败,请重试");
        }
    }

    /**
     * 获取微信小程序码
     */
    public Result getQrCode(MiniQrCodeDTO miniQrCodeDTO) {
        try {
            // 获取base64数据
            byte[] qrCode = wxMaService.getQrcodeService().createWxaCodeUnlimitBytes(
                    miniQrCodeDTO.getScene(),
                    miniQrCodeDTO.getPath(),
                    false,
                    "release",
                    miniQrCodeDTO.getWidth(),
                    false, new WxMaCodeLineColor("0", "0", "0"), false);
            return success(qrCode, "获取微信小程序码成功");
        } catch (Exception e) {
            log.error("{}", e);
            throw new ServiceException("获取微信小程序码失败");
        }
    }

    /**
     * 更新用户手机号
     */
    private void updatePhone(Long userId, String phoneNumber) {
        // 更新用户手机号
        User user = userService.selectByPrimaryKey(userId);
        user.setPhone(phoneNumber);
        userService.updateByPrimaryKeySelective(user);
    }

}
