package com.onsiteservice.miniapp.service.order;

import com.github.pagehelper.PageHelper;
import com.onsiteservice.common.order.vo.OrderPayVO;
import com.onsiteservice.common.order.vo.ServiceOrderLogVO;
import com.onsiteservice.common.order.vo.ServiceOrderVO;
import com.onsiteservice.common.service.CommonSmsService;
import com.onsiteservice.constant.constant.BizConstants;
import com.onsiteservice.constant.constant.SysConstants;
import com.onsiteservice.constant.constant.SysParamConstants;
import com.onsiteservice.constant.enums.BizCodeEnum;
import com.onsiteservice.constant.enums.ServiceOrderOpSourceEnum;
import com.onsiteservice.constant.enums.ServiceOrderStatusEnum;
import com.onsiteservice.constant.enums.ServiceUserTypeEnum;
import com.onsiteservice.core.exception.ServiceException;
import com.onsiteservice.core.result.Result;
import com.onsiteservice.core.result.ResultGenerator;
import com.onsiteservice.dao.common.AbstractMapper;
import com.onsiteservice.dao.common.page.PageInfoVO;
import com.onsiteservice.dao.component.RecordComponent;
import com.onsiteservice.dao.mapper.pay.PayChannelWechatMapper;
import com.onsiteservice.dao.mapper.service.*;
import com.onsiteservice.dao.mapper.sys.SysParamMapper;
import com.onsiteservice.dao.mapper.sys.SysUserMapper;
import com.onsiteservice.dao.mapper.user.UserMapper;
import com.onsiteservice.entity.address.ServiceAddress;
import com.onsiteservice.entity.category.ServiceCategory;
import com.onsiteservice.entity.category.ServiceSubclass;
import com.onsiteservice.entity.order.ServiceOrder;
import com.onsiteservice.entity.service.ServiceOrderImg;
import com.onsiteservice.entity.service.ServiceOrderLog;
import com.onsiteservice.entity.sys.SysParam;
import com.onsiteservice.entity.user.User;
import com.onsiteservice.miniapp.controller.order.dto.CancelServiceOrderDTO;
import com.onsiteservice.miniapp.controller.order.dto.PageServiceOrderDTO;
import com.onsiteservice.miniapp.controller.order.dto.ReserveServiceOrderDTO;
import com.onsiteservice.miniapp.controller.order.vo.ServiceOrderDefDetailVO;
import com.onsiteservice.miniapp.mapper.order.ServiceOrderBizMapper;
import com.onsiteservice.miniapp.service.weixin.pay.WechatNativePay;
import com.onsiteservice.service.order.ServiceOrderService;
import com.onsiteservice.util.AttrCopyUtils;
import com.onsiteservice.util.RandomUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.amqp.core.AmqpTemplate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import tk.mybatis.mapper.entity.Condition;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

@Service
@Slf4j
@Transactional(rollbackFor = Exception.class)
public class ServiceOrderBizService extends AbstractMapper<ServiceOrder> {

    /**
     * 短信相关
     */
    @Autowired
    private CommonSmsService commonSmsService;

    @Resource
    private ServiceOrderMapper serviceOrderMapper;
    @Resource
    private ServiceOrderImgMapper serviceOrderImgMapper;
    @Resource
    private ServiceOrderLogMapper serviceOrderLogMapper;
    @Resource
    private ServiceOrderService serviceOrderService;
    @Resource
    private ServiceAddressMapper serviceAddressMapper;
    @Resource
    private ServiceSubclassMapper serviceSubclassMapper;
    @Resource
    private ServiceCategoryMapper serviceCategoryMapper;
    @Resource
    private UserMapper userMapper;
    @Autowired
    private RecordComponent recordComponent;
    @Resource
    private PayChannelWechatMapper payChannelWechatMapper;
    @Resource
    private SysParamMapper sysParamMapper;
    @Resource
    private SysUserMapper sysUserMapper;
    @Resource
    private AmqpTemplate amqpTemplate;
    @Resource
    private ServiceOrderBizMapper serviceOrderBizMapper;


    public ServiceOrderVO selectById(Long id, Long userId) {
        log.info("order selectById id: {}, userId: {}", id, userId);

        // 订单信息
        Condition c = new Condition(ServiceOrder.class);
        c.createCriteria().andEqualTo("id", id);
        List<ServiceOrder> serviceOrderList = this.selectByCondition(c);
        if (CollectionUtils.isEmpty(serviceOrderList)) {
            throw new ServiceException(BizCodeEnum.SERVICE_ORDER_NOT_EXIST);
        }

        // 基础订单信息
        ServiceOrderVO serviceOrderVO = buildServiceOrderVO(serviceOrderList).get(0);

        // 需求图片
        Condition c1 = new Condition(ServiceOrderImg.class);
        c1.createCriteria().andEqualTo(BizConstants.OrderConstants.ORDER_ID, serviceOrderVO.getId());
        List<ServiceOrderImg> serviceOrderImgList = serviceOrderImgMapper.selectByCondition(c1);
        List<String> serviceOrderImgUrlList = serviceOrderImgList.parallelStream().map(ServiceOrderImg::getUrl).collect(Collectors.toList());
        serviceOrderVO.setDemandImgUrls(serviceOrderImgUrlList);

        // 流程信息
        Condition c2 = new Condition(ServiceOrderLog.class);
        c2.createCriteria().andEqualTo(BizConstants.OrderConstants.ORDER_ID, id);
        List<ServiceOrderLog> serviceOrderLogList = serviceOrderLogMapper.selectByCondition(c2);
        List<ServiceOrderLogVO> serviceOrderLogVOList = serviceOrderLogList.parallelStream()
                .map(e -> AttrCopyUtils.copy(e, new ServiceOrderLogVO()))
                .sorted(Comparator.comparing(ServiceOrderLogVO::getCreateTime).reversed()).collect(Collectors.toList());
        serviceOrderVO.setProcess(serviceOrderLogVOList);

        return serviceOrderVO;
    }


    public Result<PageInfoVO> getPage(PageServiceOrderDTO dto, Long userId) {
        log.info("order getPage dto: {}, userId: {}", dto, userId);

        User user = userMapper.selectByPrimaryKey(userId);
        if (Objects.isNull(user)
                || (!user.getRoleType().equals(ServiceUserTypeEnum.ADMIN.getId()) && !user.getRoleType().equals(ServiceUserTypeEnum.VALUATOR.getId()))) {
            throw new ServiceException(BizCodeEnum.NO_AUTH);
        }

        if (!BizConstants.OrderConstants.FRONT_BACK_ORDER_STATUS_MAP.containsKey(dto.getOrderStatus())) {
            throw new ServiceException(BizCodeEnum.SERVICE_ORDER_STATUS_ERROR);
        }

        PageHelper.startPage(dto.getPage(), dto.getSize());
        List<ServiceOrder> serviceOrderList = serviceOrderMapper.selectServiceOrderPage(
                BizConstants.OrderConstants.FRONT_BACK_ORDER_STATUS_MAP.get(dto.getOrderStatus()),
                dto.getMonth(),
                dto.getKeyWord(),
                user.getRoleType(),
                userId);

        PageInfoVO vo = new PageInfoVO(serviceOrderList);
        vo.setList(buildServiceOrderVO(serviceOrderList));

        return ResultGenerator.success(vo);
    }


    public Result<PageInfoVO> getMinePage(PageServiceOrderDTO dto, Long userId) {
        log.info("order getMinePage dto: {}, userId: {}", dto, userId);

        if (!BizConstants.OrderConstants.FRONT_BACK_ORDER_STATUS_MAP.containsKey(dto.getOrderStatus())) {
            throw new ServiceException(BizCodeEnum.SERVICE_ORDER_STATUS_ERROR);
        }

        PageHelper.startPage(dto.getPage(), dto.getSize());
        List<ServiceOrder> serviceOrderList = serviceOrderMapper.selectServiceOrderMinePage(
                BizConstants.OrderConstants.FRONT_BACK_ORDER_STATUS_MAP.get(dto.getOrderStatus()),
                dto.getMonth(),
                dto.getKeyWord(),
                userId);

        PageInfoVO vo = new PageInfoVO(serviceOrderList);
        vo.setList(buildServiceOrderVO(serviceOrderList));

        return ResultGenerator.success(vo);
    }


    public int reserve(ReserveServiceOrderDTO dto, Long userId, ServiceOrderOpSourceEnum sourceEnum) {
        log.info("order reserve dto: {}, userId: {}", dto, userId);

        ServiceSubclass serviceSubclass = serviceSubclassMapper.selectByPrimaryKey(dto.getSubclassId());
        if (Objects.isNull(serviceSubclass)) {
            throw new ServiceException(BizCodeEnum.SERVICE_SUBCLASS_NOT_EXIST);
        }

        ServiceCategory serviceCategory = serviceCategoryMapper.selectByPrimaryKey(serviceSubclass.getCategoryId());

        ServiceAddress serviceAddress = serviceAddressMapper.selectByPrimaryKey(dto.getAddressId());
        if (Objects.isNull(serviceAddress)) {
            throw new ServiceException(BizCodeEnum.SERVICE_ADDRESS_NOT_EXIST);
        }

        ServiceOrder serviceOrder = AttrCopyUtils.copy(dto, new ServiceOrder());
        // 账户号
        serviceOrder.setAccountNo(userId);
        // 订单号
        serviceOrder.setOrderNo(RandomUtils.orderNum());
        // 客户名
        serviceOrder.setName(serviceAddress.getName());
        // 客户手机号
        serviceOrder.setPhone(serviceAddress.getPhone());
        // 客户地址
        serviceOrder.setAddress(serviceAddress.getNamePath() + serviceAddress.getAddress());
        // 创建人
        serviceOrder.setCreateBy(userId);
        // 服务名
        serviceOrder.setServiceName(
                String.format(BizConstants.FormatConstants.CATEGORY_SUBCLASS_SERVICE_NAME,
                        serviceCategory.getServiceName(),
                        serviceSubclass.getServiceName()));

        int result = this.insertSelective(serviceOrder);

        recordComponent.recordServiceOrderImg(serviceOrder, dto.getUrls());

        // 记录流程
        if (result == 1) {
            recordComponent.recordProcess(serviceOrder.getId(), ServiceOrderStatusEnum.RESERVE.getStatus(), ServiceOrderStatusEnum.RESERVE.getMsg(), sourceEnum, null, null, null);
        }

        return result;
    }

    public void paySuccess(String orderNo, String info) {
        log.info("支付成功 orderNo: {}, info: {}", orderNo, info);

        var order = selectOneByProperty("orderNo", orderNo);

        order.setOrderStatus(ServiceOrderStatusEnum.PAY.getStatus());

        int result = serviceOrderMapper.updateByPrimaryKeySelective(order);


        // 通知客服
        var logCondition = new ServiceOrderLog();
        logCondition.setOrderId(order.getId());
        logCondition.setProcessId(ServiceOrderStatusEnum.SEND.getStatus());
        var orderLog = serviceOrderLogMapper.selectOne(logCondition);
        if (orderLog != null) {
            if (ServiceOrderOpSourceEnum.WEB.getId().equals(orderLog.getSource())) {
                var sysUser = sysUserMapper.selectByPrimaryKey(orderLog.getHostId());
                if (StringUtils.isNotEmpty(sysUser.getPhone())) {
                    try {
                        commonSmsService.paid(sysUser.getPhone());
                        amqpTemplate.convertAndSend(SysConstants.Queue.ADMIN_ORDER, Pair.of(sysUser.getId(), 1));
                    } catch (Exception e) {
                        e.printStackTrace();
                        log.error("订单{} 支付成功短信发送失败", orderNo);
                    }
                }
            }
            if (ServiceOrderOpSourceEnum.MINI_APP.getId().equals(orderLog.getSource())) {
                var user = userMapper.selectByPrimaryKey(orderLog.getHostId());
                try {
                    commonSmsService.paid(user.getPhone());
                } catch (Exception e) {
                    e.printStackTrace();
                    log.error("订单{} 支付成功短信发送失败", orderNo);
                }
            }
        }
        if (result == 1) {
            recordComponent.recordProcess(order.getId(), ServiceOrderStatusEnum.PAY.getStatus(), ServiceOrderStatusEnum.PAY.getMsg(), ServiceOrderOpSourceEnum.MINI_APP, null, null, null);
        }

    }


    public int cancelOrder(CancelServiceOrderDTO dto, Long userId, ServiceOrderOpSourceEnum sourceEnum) {
        log.info("cancelOrder dto: {}, userId: {}", dto, userId);

        ServiceOrder serviceOrder = serviceOrderService.checkOrder(dto.getId(), userId);
        if (serviceOrder.getOrderStatus() >= ServiceOrderStatusEnum.PAY.getStatus()) {
            throw new ServiceException(BizCodeEnum.SERVICE_ORDER_CANNOT_CANCEL);
        }

        serviceOrderService.disableServiceValuatorAssign(serviceOrder, userId, null);

        serviceOrder.setOrderStatus(ServiceOrderStatusEnum.CANCEL.getStatus());
        serviceOrder.setModifyBy(userId);

        int result = serviceOrderMapper.updateByPrimaryKeySelective(serviceOrder);

        if (result == 1) {
            recordComponent.recordProcess(serviceOrder.getId(), ServiceOrderStatusEnum.CANCEL.getStatus(), ServiceOrderStatusEnum.CANCEL.getMsg(), sourceEnum, null, null, null);
        }

        return result;
    }

    /**
     * 生成支付参数
     */
    public OrderPayVO pay(Long orderId, Long userId, HttpServletRequest request) {
        if (userId == null) {
            throw new ServiceException("用户未登录");
        }
        var order = selectByPrimaryKey(orderId);
        if (!order.getAccountNo().equals(userId)) {
            throw new ServiceException("用户错误");
        }
        if (new Date().compareTo(order.getCancelTime()) > 0) {
            throw new ServiceException("支付超时");
        }
        if (!order.getOrderStatus().equals(ServiceOrderStatusEnum.SEND.getStatus())) {
            throw new ServiceException("不能支付的订单状态");
        }
        var payChannel = payChannelWechatMapper.selectAll().get(0);
        WechatNativePay wechatNativePay = new WechatNativePay(payChannel.getAppid(), payChannel.getMchid(), payChannel.getWechatKey());
        String notifyUrl = sysParamMapper.selectOne(SysParam.builder().code(SysParamConstants.MINI_WECHAT_PAY_NOTIFY_URL).build()).getValue();

        var user = userMapper.selectByPrimaryKey(userId);
        try {
            var params = wechatNativePay.getPayParam("服务预约", order.getOrderNo(), order.getPrice(), notifyUrl,
                    user.getOpenId(), request);
            return new OrderPayVO(params);
        } catch (Exception e) {
            e.printStackTrace();
            throw new ServiceException(e.getMessage());
        }
    }


    /**
     * 未派单订单 用户可实时退款
     * 已派单订单 需要管理员审批
     */
    /*public int refundOrder(RefundServiceOrderDTO dto, Long userId) {
        log.info("refundOrder dto: {}, userId: {}", dto, userId);

        ServiceOrder serviceOrder = checkOrder(dto.getId(), userId);
        if (!serviceOrder.getOrderStatus().equals(ServiceOrderStatusEnum.PAY.getStatus())) {
            throw new ServiceException(BizCodeEnum.SERVICE_ORDER_CANNOT_REFUND);
        }

        // 已申请退款
        serviceOrder.setOrderStatus(ServiceOrderStatusEnum.REFUND_APPLY.getStatus());
        int result = serviceOrderMapper.updateByPrimaryKeySelective(serviceOrder);

        try {
            //  调用微信接口退款 ??

            //  通知web
        } catch (Exception e) {

        }

        return result;
    }*/
    private List<ServiceOrderVO> buildServiceOrderVO(List<ServiceOrder> serviceOrderList) {
        List<Integer> subclassIdList = serviceOrderList.parallelStream().map(ServiceOrder::getSubclassId).collect(Collectors.toList());
        List<ServiceSubclass> serviceSubclassList = serviceSubclassMapper.selectByIdList(subclassIdList);
        Map<Integer, ServiceSubclass> serviceSubclassMap = serviceSubclassList.parallelStream().collect(Collectors.toMap(ServiceSubclass::getId, Function.identity()));

        return serviceOrderList.parallelStream().map(serviceOrder -> {
            ServiceOrderVO serviceOrderVO = new ServiceOrderVO();
            serviceOrderVO.setId(serviceOrder.getId());
            serviceOrderVO.setOrderNo(serviceOrder.getOrderNo());
            serviceOrderVO.setServiceName(serviceOrder.getServiceName());
            serviceOrderVO.setServiceSubclassName(serviceOrder.getServiceName().split("-")[1]);
            serviceOrderVO.setName(serviceOrder.getName());
            serviceOrderVO.setPhone(serviceOrder.getPhone());
            serviceOrderVO.setAddress(serviceOrder.getAddress());
            serviceOrderVO.setExpectArrivalTime(serviceOrder.getExpectArrivalTime());
            serviceOrderVO.setNum(serviceOrder.getNum());
            serviceOrderVO.setDemandDesc(serviceOrder.getDemandDesc());
            serviceOrderVO.setRemark(serviceOrder.getRemark());
            serviceOrderVO.setCreateTime(serviceOrder.getCreateTime());
            serviceOrderVO.setOrderStatus(String.valueOf(serviceOrder.getOrderStatus()));
            serviceOrderVO.setPrice(serviceOrder.getPrice());
            // 剩余时间
            if (serviceOrder.getOrderStatus().equals(ServiceOrderStatusEnum.SEND.getStatus())) {
                long left = Math.max(serviceOrder.getCancelTime().getTime() - System.currentTimeMillis(), 0) / 1000;
                serviceOrderVO.setLeftTime(left);
            }

            serviceOrderVO.setSubclassImg(serviceSubclassMap.getOrDefault(serviceOrder.getSubclassId(), new ServiceSubclass()).getImg());

            return serviceOrderVO;
        }).collect(Collectors.toList());
    }


    private boolean checkRoleType(User user) {
        for (ServiceUserTypeEnum userTypeEnum : ServiceUserTypeEnum.values()) {
            if (user.getRoleType().equals(userTypeEnum.getId())) {
                return true;
            }
        }

        return false;
    }


    public ServiceOrderDefDetailVO selectDefDetail(Long userId) {
        ServiceOrderDefDetailVO serviceOrderDefDetailVO = ServiceOrderDefDetailVO.builder().build();

        Condition c = new Condition(ServiceAddress.class);
        c.createCriteria().andEqualTo(BizConstants.UserConstants.ACCOUNT_NO, userId)
                .andEqualTo("def", true)
                .andEqualTo(BizConstants.CommonConstants.DELETED, false);
        List<ServiceAddress> serviceAddressList = serviceAddressMapper.selectByCondition(c);
        if (!CollectionUtils.isEmpty(serviceAddressList)) {
            log.info("order selectDefDetail userId: {}", userId);
            serviceOrderDefDetailVO = AttrCopyUtils.copy(serviceAddressList.get(0), new ServiceOrderDefDetailVO());
        }

        // 温馨提示
        String prompt = sysParamMapper.selectOne(SysParam.builder().code(SysParamConstants.USER_PROMPT).build()).getValue();
        serviceOrderDefDetailVO.setTips(prompt);

        return serviceOrderDefDetailVO;
    }


    public String agreement() {
        return sysParamMapper.selectOne(SysParam.builder().code(SysParamConstants.AGREEMENT).build()).getValue();
    }


    /**
     * 获取各个状态（前端状态）下的待办数量
     * @param userId
     * @return
     */
    public Map<Integer, Integer> getTodoNum(Long userId) {
        User user = userMapper.selectByPrimaryKey(userId);
        if(!ServiceUserTypeEnum.ADMIN.getId().equals(user.getRoleType())) {
            return Map.of(1, 0, 3, 0, 4, 0, 5, 0, 6, 0);
        }

        return serviceOrderBizMapper.getTodoNum();
    }

}
