package com.onsiteservice.constant.enums.convert;


import com.onsiteservice.constant.enums.core.BaseEnum;
import org.springframework.core.convert.converter.Converter;
import org.springframework.core.convert.converter.ConverterFactory;
import org.springframework.stereotype.Component;

import java.util.HashMap;
import java.util.Map;
import java.util.WeakHashMap;

/**
 * @author 潘维吉
 * @date 2019-01-05 8:29
 * 统一枚举类型转换绑定  Controller层接受code统一转换成枚举类
 * 在Spring MVC和Spring Boot中，由于从客户端接收到的请求都被视为String类型，所以只能用String转枚举的converter
 * 三个泛型，S、R、T，其中S表示原类型，R表示目标类型，T是类型R的一个子类
 */
@Component
public class EnumConverterFactory implements ConverterFactory<String, BaseEnum> {

    private static final Map<Class, Converter> CONVERTER_MAP = new WeakHashMap<>();

    @Override
    public <T extends BaseEnum> Converter<String, T> getConverter(Class<T> targetType) {
        if (!targetType.isEnum()) {
            throw new UnsupportedOperationException("只支持转换到枚举类型");
        }
        Converter result = CONVERTER_MAP.get(targetType);
        if (result == null) {
            result = new IntegerStrToEnum<T>(targetType);
            CONVERTER_MAP.put(targetType, result);
        }
        return result;
    }

    class IntegerStrToEnum<T extends BaseEnum> implements Converter<String, T> {
        private final Class<T> enumType;
        private Map<String, T> enumMap = new HashMap<>();

        public IntegerStrToEnum(Class<T> enumType) {
            this.enumType = enumType;
            T[] enums = enumType.getEnumConstants();

            for (T e : enums) {
                enumMap.put(e.getCode() + "", e);
            }
        }

        @Override
        public T convert(String code) {
            T result = enumMap.get(code);
            if (result == null) {
                throw new IllegalArgumentException("No element matches " + code);
            }
            return result;
        }
    }

}
