package com.onsiteservice.util.tree;

import com.alibaba.fastjson.JSON;
import lombok.Getter;
import lombok.Setter;

import java.util.ArrayList;
import java.util.List;
import java.util.Stack;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author 潘维吉
 * @date 2020/4/13 9:12
 * @email 406798106@qq.com
 * @description 生成树状结构工具类
 * 提供通用树结构的聚合/平铺方法
 */
public class TreeUtils {

    /**
     * 聚合树结构
     *
     * @param list          节点列表结构
     * @param loadKey       节点唯一key读取 接收一个节点 返回节点的唯一key
     * @param loadParentKey 节点父节点key读取 接收一个节点 返回节点的父节点key
     * @param write         节点子项写入函数 接收待写入节点与节点子项 负责将子节点写入
     * @param <T>           节点对象
     * @param <R>           节点唯一key对象
     * @return 树结构
     */
    public static <T, R> List<T> getTree(List<T> list, Function<T, R> loadKey, Function<T, R> loadParentKey, BiConsumer<T, List<T>> write) {
        List<T> root = list.stream().filter(o -> loadParentKey.apply(o) == null).collect(Collectors.toList());
        Stack<T> stack = new Stack<>();
        root.forEach(stack::push);
        while (!stack.isEmpty()) {
            T o = stack.pop();
            R key = loadKey.apply(o);
            List<T> children = list.stream()
                    .filter(k -> key.equals(loadParentKey.apply(k)))
                    .collect(Collectors.toList());
            write.accept(o, children);
            if (children.size() > 0) {
                children.forEach(stack::push);
            }
        }
        return root;
    }

    /**
     * 平铺树结构
     *
     * @param root              节点树结构
     * @param loadChildrenNodes 加载树的子节点列表函数 接收一个节点 返回节点的子结构
     * @param <T>               树节点对象
     * @return 平铺结构
     */
    public static <T> List<T> getTileTree(List<T> root, Function<T, List<T>> loadChildrenNodes) {
        List<T> list = new ArrayList<>();
        traversing(root, loadChildrenNodes, list::add);
        return list;
    }

    /**
     * 遍历树结构
     *
     * @param root              节点树结构
     * @param loadChildrenNodes 加载树的子节点列表函数 接收一个节点 返回节点的子结构
     * @param behavior          遍历到的节点行为
     * @param <T>               树节点对象
     */
    public static <T> void traversing(List<T> root, Function<T, List<T>> loadChildrenNodes, Consumer<T> behavior) {
        Stack<T> stack = new Stack<>();
        root.forEach(stack::push);
        while (!stack.isEmpty()) {
            T o = stack.pop();
            behavior.accept(o);
            List<T> children = loadChildrenNodes.apply(o);
            if (children != null && children.size() > 0) {
                children.forEach(stack::push);
            }
        }
    }

    /**
     * 打印树信息
     *
     * @param list              树结构列表
     * @param loadChildrenNodes 加载树的子节点列表函数 接收一个节点 返回节点的子结构
     * @param <T>               树节点对象
     */
    public static <T> void printTree(List<T> list, Function<T, List<T>> loadChildrenNodes) {
        System.out.println("---------- Tree Nodes Print ----------");
        traversing(list, loadChildrenNodes, System.out::println);
        System.out.println("---------- Tree Nodes Print ----------");
    }

    /**
     * 根据需求自定义组装Node VO对象
     */
    @Getter
    @Setter
    public static class Node {
        private Integer id;
        private Integer parentId;
        private String name;
        private String treePath;
        private String otherProperty; //其他的属性  可选
        private List<Node> children;

        public Node(Integer id, String name, Integer parentId) {
            this.id = id;
            this.parentId = parentId;
            this.name = name;
        }

        public Node(Integer id, String name, Integer parentId, String otherProperty) {
            this.id = id;
            this.parentId = parentId;
            this.name = name;
            this.otherProperty = otherProperty;
        }
    }

    public static void main(String[] args) {
        List<Node> listNodes = new ArrayList<>();
        listNodes.add(new Node(1, "根节点1", null, "其他属性"));
        listNodes.add(new Node(2, "根节点2", null, "其他属性"));
        listNodes.add(new Node(3, "根节点3", null, "其他属性"));
        listNodes.add(new Node(4, "1-1", 1, "其他属性"));
        listNodes.add(new Node(5, "1-2", 1, "其他属性"));
        listNodes.add(new Node(6, "2-1", 2, "其他属性"));
        listNodes.add(new Node(7, "3-1", 3, "其他属性"));
        listNodes.add(new Node(8, "1-1-1", 4, "其他属性"));
        listNodes.add(new Node(9, "1-1-2", 4, "其他属性"));
        System.out.println("基础数据");
        //printTree(listNodes, Node::getChildren);
        System.out.println(JSON.toJSONString(listNodes));

        System.out.println("动态生成聚合TreePath路径");
        TreePathUtils.genPath(listNodes);
        for (Node node : listNodes) {
            System.out.println(node.getId() + " -> " + node.getTreePath());
        }

        // 聚合
        System.out.println("聚合Tree数据");
        List<Node> treeNodes = getTree(listNodes, Node::getId, Node::getParentId, Node::setChildren);
        //printTree(treeNodes, Node::getChildren);
        System.out.println(JSON.toJSONString(treeNodes));

        // 平铺
        //System.out.println("平铺Tree数据");
        //getTileTree(treeNodes, Node::getChildren);
        //printTree(listNodes, Node::getChildren);
        //System.out.println(JSON.toJSONString(listNodes));
    }
}

