package com.onsiteservice.util;

import com.google.zxing.*;
import com.google.zxing.client.j2se.BufferedImageLuminanceSource;
import com.google.zxing.client.j2se.MatrixToImageWriter;
import com.google.zxing.common.BitMatrix;
import com.google.zxing.common.HybridBinarizer;
import com.google.zxing.qrcode.decoder.ErrorCorrectionLevel;
import com.onsiteservice.util.aliyun.OssUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.IOUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.core.io.ClassPathResource;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import javax.imageio.ImageIO;
import java.awt.*;
import java.awt.geom.RoundRectangle2D;
import java.awt.image.BufferedImage;
import java.io.*;
import java.util.Hashtable;
import java.util.Map;

/**
 * @author 潘维吉
 * @date 2019-11-20 11:21
 * 二维码、条形码生成与解析的工具类
 */
@ConditionalOnProperty(prefix = "aliyun.oss", name = {"enabled"}, matchIfMissing = false)
@Service
@Slf4j
public class QRCodeUtils {

    private static final String CHARSET = "utf-8";
    private static final String FORMAT_NAME = "JPG";
    // 二维码尺寸
    private static final int QR_CODE_SIZE = 300;
    // LOGO宽度
    private static final int WIDTH = 60;
    // LOGO高度
    private static final int HEIGHT = 60;
    // OSS路径
    private static final String OSS_PATH = "qr_code/";
    // 二维码图片后缀
    private static final String IMAGE_SUFFIX = ".jpg";

    @Resource
    private OssUtils ossUtils;
    // 维护一个本类的静态变量
    public static QRCodeUtils qrCodeUtils;

    // 初始化的时候，赋值给静态的本类变量
    @PostConstruct
    public void init() {
        qrCodeUtils = this;
        qrCodeUtils.ossUtils = this.ossUtils;
    }

    public void encode(String content, OutputStream output) throws Exception {
        BufferedImage image = QRCodeUtils.createImage(content, null, false, QR_CODE_SIZE, QR_CODE_SIZE);
        ImageIO.write(image, FORMAT_NAME, output);
    }

    public void encode(String content, String imgPath, String destPath) throws Exception {
        QRCodeUtils.encode(content, imgPath, destPath, false, QR_CODE_SIZE, QR_CODE_SIZE);
    }

    // 不带logo图的二维码
    public void encode(String content, String destPath, boolean needCompress) throws Exception {
        QRCodeUtils.encode(content, null, destPath, needCompress, QR_CODE_SIZE, QR_CODE_SIZE);
    }

    public void encode(String content, String destPath) throws Exception {
        QRCodeUtils.encode(content, null, destPath, false, QR_CODE_SIZE, QR_CODE_SIZE);
    }

    // 生成二维码并上传到oss
    public String encodeOss(String content) throws Exception {
        return encodeOss(content, QR_CODE_SIZE, QR_CODE_SIZE);
    }

    // 生成二维码并上传到oss 自定义二维码大小
    public String encodeOss(String content, int width, int height) throws Exception {
        BufferedImage image = QRCodeUtils.createImage(content, null, false, width, height);
        InputStream in = QRCodeUtils.bufferedImageToInputStream(image);
        //上传file二维码到阿里云
        return qrCodeUtils.ossUtils.uploadFile(
                OSS_PATH + UUIDUtils.getFlowID() + IMAGE_SUFFIX, IOUtils.toByteArray(in));
    }

    // 生成二维码并上传到oss 自定义插入文字
    public String encodeText(String content, Map map, int width, int height) throws Exception {
        BufferedImage image = QRCodeUtils.createImage(content, null, false, width, height);
        Hashtable hints = new Hashtable();
        // 指定纠错等级
        hints.put(EncodeHintType.ERROR_CORRECTION, ErrorCorrectionLevel.Q);
        // 指定编码格式
        hints.put(EncodeHintType.CHARACTER_SET, "UTF-8");
        // 生成二维码
        BitMatrix bitMatrix = new MultiFormatWriter().encode(content,
                BarcodeFormat.QR_CODE, width, height, hints);
        //重新生成一个画布
        Graphics2D graphics = image.createGraphics();
        //设置背景颜色
        graphics.setColor(Color.white);
        graphics.fillRect(0, 0, width, height);
        graphics.drawImage(MatrixToImageWriter.toBufferedImage(bitMatrix), -height / 2, 5, null);
        //加入文字  注意linux上添加设置的字体库 不添加出现中文乱码
        graphics.setFont(new Font("微软雅黑", Font.BOLD, width / 30));
        graphics.setColor(Color.black);
        graphics.drawString("名称: " + map.get("name"), width / 2, height / 4 + height / 10);
        graphics.drawString("编号: " + map.get("code"), width / 2, height / 3 + height / 5);
        graphics.drawString("位置: " + map.get("location"), width / 2, height / 2 + height / 5);
        //关闭图片
        graphics.dispose();
        InputStream in = QRCodeUtils.bufferedImageToInputStream(image);
        //上传file二维码到阿里云
        return qrCodeUtils.ossUtils.uploadFile(
                OSS_PATH + UUIDUtils.getFlowID() + IMAGE_SUFFIX, IOUtils.toByteArray(in));
    }

    /**
     * 生成二维码，并保存到磁盘上
     *
     * @param content      地址链接
     * @param imgPath      logo图片
     * @param destPath     二维码保存目标路径
     * @param needCompress 是否压缩
     * @return oss二维码图片路径
     * @throws Exception
     */
    public static String encode(String content, String imgPath, String destPath, boolean needCompress, int qrWidth, int qrHeight) throws Exception {
        BufferedImage image = QRCodeUtils.createImage(content, imgPath, needCompress, qrWidth, qrHeight);
        String os = System.getProperty("os.name");
//       os = "Linux 10";
        InputStream in = null;
        if (os.indexOf("Windows") > -1) {
            ImageIO.write(image, FORMAT_NAME, new File(destPath));
            in = new BufferedInputStream(new FileInputStream(destPath));
        } else {
            //inputStream转outPutStream
            InputStream inputStream = new ClassPathResource(destPath).getInputStream();
            ByteArrayOutputStream swapStream = new ByteArrayOutputStream();
            int ch;
            while ((ch = inputStream.read()) != -1) {
                swapStream.write(ch);
            }
            boolean flag = ImageIO.write(image, FORMAT_NAME, swapStream);
            ClassPathResource destPathClass = new ClassPathResource(destPath);
            in = destPathClass.getInputStream();

        }
        return destPath;
    }

    /**
     * 生成二维码
     *
     * @param content      源内容
     * @param imgPath      生成二维码保存的路径
     * @param needCompress 是否要压缩
     * @return 返回二维码图片
     * @throws Exception
     */
    private static BufferedImage createImage(String content, String imgPath, boolean needCompress, int qrWidth, int qrHeight) throws Exception {
        Hashtable hints = new Hashtable();
        hints.put(EncodeHintType.ERROR_CORRECTION, ErrorCorrectionLevel.H);
        hints.put(EncodeHintType.CHARACTER_SET, CHARSET);
        hints.put(EncodeHintType.MARGIN, 1);
        BitMatrix bitMatrix = new MultiFormatWriter().encode(content, BarcodeFormat.QR_CODE, qrWidth, qrHeight, hints);
        int width = bitMatrix.getWidth();
        int height = bitMatrix.getHeight();
        BufferedImage image = new BufferedImage(width, height, BufferedImage.TYPE_INT_RGB);
        for (int x = 0; x < width; x++) {
            for (int y = 0; y < height; y++) {
                image.setRGB(x, y, bitMatrix.get(x, y) ? 0xFF000000 : 0xFFFFFFFF);
            }
        }
        if (imgPath == null || "".equals(imgPath)) {
            return image;
        }
        // 插入和压缩图片
        QRCodeUtils.insertImage(image, imgPath, needCompress, qrWidth, qrHeight);
        return image;
    }

    /**
     * 在生成的二维码中插入图片
     *
     * @param source
     * @param imgPath
     * @param needCompress
     * @throws Exception
     */
    private static void insertImage(BufferedImage source, String imgPath, boolean needCompress, int qrWidth, int qrHeight) throws Exception {
        File file = new File(imgPath);
        if (!file.exists()) {
            System.err.println("" + imgPath + " 该文件不存在！");
            return;
        }
        Image src = ImageIO.read(new File(imgPath));
        int width = src.getWidth(null);
        int height = src.getHeight(null);
        if (needCompress) { // 压缩LOGO
            if (width > WIDTH) {
                width = WIDTH;
            }
            if (height > HEIGHT) {
                height = HEIGHT;
            }
            Image image = src.getScaledInstance(width, height, Image.SCALE_SMOOTH);
            BufferedImage tag = new BufferedImage(width, height, BufferedImage.TYPE_INT_RGB);
            Graphics g = tag.getGraphics();
            g.drawImage(image, 0, 0, null); // 绘制缩小后的图
            g.dispose();
            src = image;
        }
        // 插入LOGO
        Graphics2D graph = source.createGraphics();
        int x = (qrWidth - width) / 2;
        int y = (qrHeight - height) / 2;
        graph.drawImage(src, x, y, width, height, null);
        Shape shape = new RoundRectangle2D.Float(x, y, width, width, 6, 6);
        graph.setStroke(new BasicStroke(3f));
        graph.draw(shape);
        graph.dispose();
    }

    /**
     * 从二维码中，解析数据
     *
     * @param file 二维码图片文件
     * @return 返回从二维码中解析到的数据值
     * @throws Exception
     */
    public static String decode(File file) throws Exception {
        BufferedImage image;
        image = ImageIO.read(file);
        if (image == null) {
            return null;
        }
        BufferedImageLuminanceSource source = new BufferedImageLuminanceSource(image);
        BinaryBitmap bitmap = new BinaryBitmap(new HybridBinarizer(source));
        Result result;
        Hashtable hints = new Hashtable();
        hints.put(DecodeHintType.CHARACTER_SET, CHARSET);
        result = new MultiFormatReader().decode(bitmap, hints);
        String resultStr = result.getText();
        return resultStr;
    }

    public static String decode(String path) throws Exception {
        return QRCodeUtils.decode(new File(path));
    }

    /**
     * 将BufferedImage转换为InputStream
     *
     * @param image
     * @return
     */
    private static InputStream bufferedImageToInputStream(BufferedImage image) {
        ByteArrayOutputStream os = new ByteArrayOutputStream();
        try {
            ImageIO.write(image, FORMAT_NAME, os);
            InputStream input = new ByteArrayInputStream(os.toByteArray());
            return input;
        } catch (IOException e) {
            log.error("提示:", e);
        }
        return null;
    }
}
