package com.onsiteservice.common.annotation.limit;

import com.google.common.collect.Maps;
import com.google.common.util.concurrent.RateLimiter;
import com.onsiteservice.core.config.MvcConfig;
import com.onsiteservice.core.result.Result;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.lang.reflect.Method;
import java.util.Map;
import java.util.Objects;

import static com.onsiteservice.core.result.ResultGenerator.fail;


/**
 * @author 潘维吉
 * @date 2020/3/13 10:48
 * @email 406798106@qq.com
 * @description AOP的环绕通知来拦截注解  使用Guava RateLimiter本地限流
 * 使用了一个ConcurrentMap来保存每个请求对应的令牌桶,key是没有url请求,防止出现每个请求都会新建一个令牌桶这么会达不到限流效果
 */
@ConditionalOnProperty(prefix = "project.rate-limit", name = {"enabled"}, matchIfMissing = false)
@Slf4j
@Aspect
@Component
public class RateLimiterAspect {

    /**
     * 使用url做为key,存放令牌桶 防止每次重新创建令牌桶
     */
    private Map<String, RateLimiter> limitMap = Maps.newConcurrentMap();

    @Pointcut("@annotation(com.onsiteservice.common.annotation.limit.RateLimit)")
    public void rateLimit() {
    }

    @Around("rateLimit()")
    public Object around(ProceedingJoinPoint joinPoint) throws Throwable {
        // 获取request,response
        HttpServletRequest request = ((ServletRequestAttributes) RequestContextHolder.getRequestAttributes()).getRequest();
        HttpServletResponse response = ((ServletRequestAttributes) RequestContextHolder.getRequestAttributes()).getResponse();
        // 或者url(存在map集合的key)
        String url = request.getRequestURI();
        // 获取自定义注解
        RateLimit rateLimiter = getRateLimit(joinPoint);
        if (rateLimiter != null) {
            RateLimiter limiter = null;
            // 判断map集合中是否有创建有创建好的令牌桶
            if (!limitMap.containsKey(url)) {
                // 创建令牌桶 创建一个稳定输出令牌的RateLimiter，保证了平均每秒不超过permitsPerSecond个请求
                // 当请求到来的速度超过了permitsPerSecond，保证每秒只处理permitsPerSecond个请求
                limiter = RateLimiter.create(rateLimiter.permitsPerSecond());
                limitMap.put(url, limiter);
                log.info("请求{} , 创建令牌桶 , 容量permitsPerSecond={} 成功", url, rateLimiter.permitsPerSecond());
            }
            limiter = limitMap.get(url);
            // 获取令牌 带超时时间
            boolean acquire = limiter.tryAcquire(rateLimiter.timeout(), rateLimiter.timeunit());

            if (!acquire) {
                Result result = fail(rateLimiter.msg());
                MvcConfig.responseResult(response, result);
                return null;
            }
        }
        return joinPoint.proceed();
    }

    /**
     * 获取注解对象
     *
     * @param joinPoint 对象
     */
    private RateLimit getRateLimit(final JoinPoint joinPoint) {
        Method[] methods = joinPoint.getTarget().getClass().getDeclaredMethods();
        String name = joinPoint.getSignature().getName();
        if (!StringUtils.isEmpty(name)) {
            for (Method method : methods) {
                RateLimit annotation = method.getAnnotation(RateLimit.class);
                if (!Objects.isNull(annotation) && name.equals(method.getName())) {
                    return annotation;
                }
            }
        }
        return null;
    }

}
