package com.onsiteservice.admin.service.service;

import com.github.pagehelper.PageHelper;
import com.google.common.collect.Lists;
import com.onsiteservice.admin.controller.service.dto.PageServiceWorkerDTO;
import com.onsiteservice.admin.controller.service.dto.SaveServiceWorkerDTO;
import com.onsiteservice.admin.controller.service.dto.UpdateServiceWorkerDTO;
import com.onsiteservice.admin.controller.service.vo.ServiceWorkerKindVO;
import com.onsiteservice.admin.controller.service.vo.ServiceWorkerVO;
import com.onsiteservice.admin.controller.service.vo.WorkerInfoVO;
import com.onsiteservice.admin.mapper.service.ServiceWorkerBizMapper;
import com.onsiteservice.constant.constant.BizConstants;
import com.onsiteservice.constant.enums.BizCodeEnum;
import com.onsiteservice.constant.enums.ServiceUserTypeEnum;
import com.onsiteservice.constant.enums.ServiceWorkerEnum;
import com.onsiteservice.core.exception.ServiceException;
import com.onsiteservice.core.result.Result;
import com.onsiteservice.core.result.ResultGenerator;
import com.onsiteservice.dao.common.AbstractMapper;
import com.onsiteservice.dao.common.page.PageInfoVO;
import com.onsiteservice.dao.mapper.service.ServiceWorkerAssignMapper;
import com.onsiteservice.dao.mapper.service.ServiceWorkerMapper;
import com.onsiteservice.dao.mapper.user.UserMapper;
import com.onsiteservice.entity.service.ServiceValuatorAssign;
import com.onsiteservice.entity.service.ServiceWorker;
import com.onsiteservice.entity.service.ServiceWorkerAssign;
import com.onsiteservice.entity.user.User;
import com.onsiteservice.util.AttrCopyUtils;
import com.onsiteservice.util.DateUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import tk.mybatis.mapper.entity.Condition;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

@Service
@Slf4j
@Transactional(rollbackFor = Exception.class)
public class ServiceWorkerService extends AbstractMapper<ServiceWorker> {

    @Resource
    private ServiceWorkerMapper serviceWorkerMapper;

    @Resource
    private ServiceWorkerBizMapper serviceWorkerBizMapper;
    @Resource
    private UserMapper userMapper;
    @Resource
    private ServiceWorkerAssignMapper serviceWorkerAssignMapper;

    /**
     * 分页查询列表
     */
    public Result<PageInfoVO<ServiceWorkerVO>> getPage(PageServiceWorkerDTO dto) {
        log.info("worker getPage dto: {}", dto);

        PageHelper.startPage(dto.getPage(), dto.getSize());
        List<ServiceWorker> serviceWorkerList = serviceWorkerBizMapper.selectServiceWorkerPage(dto);
        List<ServiceWorkerVO> serviceWorkerVOList = serviceWorkerList.parallelStream().map(e -> AttrCopyUtils.copy(e, new ServiceWorkerVO())).collect(Collectors.toList());

        return ResultGenerator.success(new PageInfoVO<>(serviceWorkerVOList));
    }

    public ServiceWorkerVO selectById(Long id) {
        log.info("worker selectById id: {}", id);

        ServiceWorker serviceWorker = this.selectByPrimaryKey(id);
        if (Objects.isNull(serviceWorker)) {
            throw new ServiceException(BizCodeEnum.SERVICE_WORKER_NOT_EXIST);
        }

        return AttrCopyUtils.copy(serviceWorker, new ServiceWorkerVO());
    }

    public int save(SaveServiceWorkerDTO dto, Long userId) {
        log.info("worker save dto: {}, userId: {}", dto, userId);

        // checkAuth(userId);

        Condition c = new Condition(ServiceWorker.class);
        c.createCriteria().andEqualTo("phone", dto.getPhone());
        List<ServiceWorker> serviceWorkerList = this.selectByCondition(c);
        if (!CollectionUtils.isEmpty(serviceWorkerList)) {
            throw new ServiceException(BizCodeEnum.SERVICE_WORKER_HAS_EXIST);
        }

        return this.insertSelective(AttrCopyUtils.copy(dto, new ServiceWorker()));
    }

    public int update(UpdateServiceWorkerDTO dto, Long userId) {
        log.info("worker update dto: {}, userId: {}", dto, userId);

        // checkAuth(userId);

        return this.updateByPrimaryKeySelective(AttrCopyUtils.copy(dto, new ServiceWorker()));
    }

    public int delete(Long id, Long userId) {
        log.info("worker delete id: {}, userId: {}", id, userId);

        var  condition = new ServiceWorkerAssign();
        condition.setWorkerId(id);
        int assignCount = serviceWorkerAssignMapper.selectCount(condition);
        if(assignCount > 0) {
            throw new ServiceException("不允许删除已经指派过的维修工");
        }
        return this.deleteByPrimaryKey(id);
    }


    public List<ServiceWorkerKindVO> kind() {
        return Arrays.stream(ServiceWorkerEnum.values()).parallel()
                .map(e -> ServiceWorkerKindVO.builder().id(e.getId()).name(e.getName()).build())
                .collect(Collectors.toList());
    }


    public List<WorkerInfoVO> selectByName(String name) {
        log.info("worker selectByName name: {}", name);
        List<ServiceWorker> workerList;
        Condition condition = new Condition(ServiceWorker.class);
        var criteria = condition.createCriteria();
        if (StringUtils.isNotEmpty(name)) {
            criteria.andLike("name", name);
        }
        workerList = serviceWorkerMapper.selectByCondition(condition);
        return handle(workerList);
    }


    private List<WorkerInfoVO> handle(List<ServiceWorker> serviceWorkerList) {
        if (CollectionUtils.isEmpty(serviceWorkerList)) {
            return Lists.newArrayList();
        }
        // 维修工id列表
        List<Long> workerIds = serviceWorkerList.parallelStream().map(ServiceWorker::getId).collect(Collectors.toList());
        // 已被指派的时间
        List<ServiceWorkerAssign> serviceWorkerAssignList = serviceWorkerAssignMapper.selectByDeletedAndIdListAndAssignTime(workerIds);
        Map<Long, List<ServiceWorkerAssign>> serviceWorkerAssignMap =
                serviceWorkerAssignList.parallelStream()
                        .collect(Collectors.groupingBy(ServiceWorkerAssign::getWorkerId));

        return serviceWorkerList.parallelStream().map(e1 -> {
            WorkerInfoVO serviceWorkerVO = AttrCopyUtils.copy(e1, new WorkerInfoVO());
            // 维修工已经被指派的时间
            List<ServiceWorkerAssign> assigns = serviceWorkerAssignMap.getOrDefault(e1.getId(), Lists.newArrayList());
            String dateFormat = "yyyy-MM-dd HH:mm";
            List<String> assignTimes = assigns.parallelStream()
                    .map(assign -> DateUtils.formatDate(assign.getAssignTime(), dateFormat)).collect(Collectors.toList());

            serviceWorkerVO.setAssignTime(assignTimes);
            return serviceWorkerVO;
        }).collect(Collectors.toList());
    }


}
