package com.onsiteservice.miniapp.service.category;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.onsiteservice.constant.constant.BizConstants;
import com.onsiteservice.constant.constant.RedisKeyConstants;
import com.onsiteservice.constant.enums.ServiceUserTypeEnum;
import com.onsiteservice.core.result.Result;
import com.onsiteservice.core.result.ResultGenerator;
import com.onsiteservice.dao.mapper.banner.BannerMapper;
import com.onsiteservice.dao.mapper.home.HomeNoticeMapper;
import com.onsiteservice.dao.mapper.service.ServiceAreaMapper;
import com.onsiteservice.dao.mapper.user.UserMapper;
import com.onsiteservice.entity.area.ServiceArea;
import com.onsiteservice.entity.banner.Banner;
import com.onsiteservice.entity.home.HomeNotice;
import com.onsiteservice.entity.user.User;
import com.onsiteservice.miniapp.controller.category.vo.*;
import com.onsiteservice.util.AttrCopyUtils;
import com.onsiteservice.util.tree.TreeUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import tk.mybatis.mapper.entity.Condition;

import javax.annotation.Resource;
import java.util.List;
import java.util.stream.Collectors;

@Service
@Slf4j
@Transactional(rollbackFor = Exception.class)
public class HomeCommonService {


    @Resource
    private BannerMapper bannerMapper;

    @Resource
    private HomeNoticeMapper homeNoticeMapper;

    @Resource
    private ServiceAreaMapper serviceAreaMapper;

    @Resource
    private ServiceCategoryService serviceCategoryService;

    @Autowired
    private StringRedisTemplate redisTemplate;

    @Resource
    private UserMapper userMapper;


    public Result<HomeViewVO> home(Long userId) {
        // 轮播图
        List<Banner> bannerList = bannerMapper.selectAll();
        List<BannerVO> bannerVOList = bannerList.parallelStream()
                .map(e -> AttrCopyUtils.copy(e, new BannerVO())).collect(Collectors.toList());

        // 公告
        Condition c = new Condition(HomeNotice.class);
        c.createCriteria().andEqualTo(BizConstants.CommonConstants.RELEASE_STATUS, 2);
        List<HomeNotice> noticeList = homeNoticeMapper.selectByCondition(c);
        List<HomeNoticeVO> homeNoticeVOList = noticeList.parallelStream()
                .map(e -> AttrCopyUtils.copy(e, new HomeNoticeVO())).collect(Collectors.toList());

        // 服务
        Result<List<ServiceCategoryVO>> serviceResult = serviceCategoryService.list(null);

        HomeViewVO homeViewVO = HomeViewVO.builder()
                .bannerList(bannerVOList)
                .noticeList(homeNoticeVOList)
                .serviceList(serviceResult.getData())
                .roleType(getRoleType(userId))
                .build();

        return ResultGenerator.success(homeViewVO);
    }


    public Result tree() {
        String value = redisTemplate.opsForValue().get(RedisKeyConstants.SERVICE_AREA_TREE);
        try {
            List<ServiceAreaVO> data = JSONArray.parseArray(value, ServiceAreaVO.class);
            if (!CollectionUtils.isEmpty(data)) {
                return ResultGenerator.success(data);
            }
        } catch (Exception e) {
            log.error("common tree, 转型失败 ", e);
        }

        List<ServiceArea> serviceAreaList = serviceAreaMapper.selectAll();
        List<ServiceAreaVO> serviceAreaVOList = serviceAreaList.parallelStream().map(e -> AttrCopyUtils.copy(e, new ServiceAreaVO())).collect(Collectors.toList());
        List<ServiceAreaVO> list = TreeUtils.getTree(serviceAreaVOList, ServiceAreaVO::getCode, ServiceAreaVO::getParentCode, ServiceAreaVO::setChildren);
        redisTemplate.opsForValue().set(RedisKeyConstants.SERVICE_AREA_TREE, JSON.toJSONString(list));

        return ResultGenerator.success(list);
    }

    private Integer getRoleType(Long userId) {
        if(userId == null) {
            return ServiceUserTypeEnum.USER.getId();
        }
        Condition condition = new Condition(User.class);
        var user = userMapper.selectByPrimaryKey(userId);
        return user.getRoleType();
    }


}
