package com.onsiteservice.dao.config;

import com.github.pagehelper.PageInterceptor;
import com.onsiteservice.constant.constant.Constants;
import com.onsiteservice.dao.common.Mapper;
import com.onsiteservice.dao.common.map.MapWrapperFactory;
import org.apache.ibatis.plugin.Interceptor;
import org.apache.ibatis.session.SqlSessionFactory;
import org.mybatis.spring.SqlSessionFactoryBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.io.support.PathMatchingResourcePatternResolver;
import org.springframework.core.io.support.ResourcePatternResolver;
import tk.mybatis.spring.mapper.MapperScannerConfigurer;

import javax.sql.DataSource;
import java.util.Properties;

/**
 * @author 潘维吉
 * @date 2019-09-28
 * Mybatis框架 & Mapper插件 & PageHelper分页 & 下划线的字段转化成驼峰命名等通用配置
 * 注意初始化项目时候暂时设置matchIfMissing=false避免数据源配置报错
 */
@ConditionalOnProperty(prefix = "project.mybatis", name = {"enabled"}, matchIfMissing = true)
@Configuration
public class MybatisConfig {

    public static final String BASE_PACKAGE = Constants.PACKAGE_NAME_GROUP + ".dao"; //dao独立模块基础包名
    public static final String MODEL_PACKAGE = BASE_PACKAGE + ".entity"; //Model模型所在包
    public static final String MAPPER_PACKAGE = "com.*.**.mapper"; //Mapper接口所在包
    public static final String MAPPER_XML_PATH = "classpath*:mapper/**/*.xml"; //Mapper XML文件位置

    public static final String TYPE_HANDLERS_PACKAGE = BASE_PACKAGE + ".common.handler"; //自定义Mybatis类型处理器所在包

    /**
     * 分页和xml映射配置 自动检测现有的数据源
     * PageHelper分页 & 下划线的字段转化成驼峰命名 不需要再设置sql每个字段别名
     * 将创建并注册的一个实例的SqlSessionFactory传递一个数据源作为使用输入SqlSessionFactoryBean的
     */
    // 主数据源配置
//    @Primary
//    @Qualifier("primaryDataSource")
//    @Bean(name = "primaryDataSource")
//    @ConfigurationProperties(prefix = "spring.shardingsphere.datasource.ds1")
//    public DataSource primaryDataSource() {
//        return DataSourceBuilder.create().build();
//    }

    //  @Qualifier("dataSource")
    @Bean
    public static SqlSessionFactory sqlSessionFactoryBean(
            DataSource dataSource) throws Exception {
        SqlSessionFactoryBean sqlSessionFactoryBean = new SqlSessionFactoryBean();
        sqlSessionFactoryBean.setDataSource(dataSource);
        //扫描统一配置别名 resultType不再需要指定完整的包名 多包名匹配 多个package之间可以用逗号或者分号等来进行分隔  批量命名别名时，默认命名为的别名为类名（不区别大小写）
        sqlSessionFactoryBean.setTypeAliasesPackage(MODEL_PACKAGE + " , com.*.*.controller.*.vo");
        //该配置会被MybatisAutoConfiguration读取，并调用SqlSessionFactoryBean来完成TypeHandler的初始化
        sqlSessionFactoryBean.setTypeHandlersPackage(TYPE_HANDLERS_PACKAGE);

        //配置分页插件，详情请查阅官方文档
        Properties properties = new Properties();
        //分页size尺寸为0时查询全部纪录
        properties.setProperty("pageSizeZero", "true");
        //页码<=0 查询第一页，页码>=总页数查询最后一页
        properties.setProperty("reasonable", "false");
        //该参数默认为false 设置为true时，使用RowBounds分页会进行count查询
        // properties.setProperty("rowBoundsWithCount", "true");
        //支持通过 Mapper 接口参数来传递分页参数
        properties.setProperty("supportMethodsArguments", "true");
        Interceptor interceptor = new PageInterceptor();
        interceptor.setProperties(properties);

        //添加分页插件和性能分析插件
        sqlSessionFactoryBean.setPlugins(new Interceptor[]{interceptor, new SqlExecuteInterceptor()});

        //添加XML目录扫描
        ResourcePatternResolver resolver = new PathMatchingResourcePatternResolver();
        sqlSessionFactoryBean.setMapperLocations(resolver.getResources(MAPPER_XML_PATH));

        SqlSessionFactory sqlSessionFactory = sqlSessionFactoryBean.getObject();
        //mybatis自动将SQL中查出来的带下划线的字段，转换为驼峰标志
        sqlSessionFactory.getConfiguration().setMapUnderscoreToCamelCase(true);
        //自定义多表联查map类型下划线的字段，转换为驼峰标志
        sqlSessionFactory.getConfiguration().setObjectWrapperFactory(new MapWrapperFactory());

        return sqlSessionFactory;
    }

    /**
     * mapper自动扫描和通用Mapper配置
     * 默认myBatis-spring-boot-starter将搜索标有@Mapper注解的映射器 可使用指定自定义注解@MapperScan或文件目录以进行扫描
     */
    @Bean
    public static MapperScannerConfigurer mapperScannerConfigurer() {
        MapperScannerConfigurer mapperScannerConfigurer = new MapperScannerConfigurer();
        mapperScannerConfigurer.setSqlSessionFactoryBeanName("sqlSessionFactoryBean");
        //统一设置mapper扫描 相当于Application.java里面设置扫描位置@MapperScan(basePackages="com.dao.mapper")
        mapperScannerConfigurer.setBasePackage(MAPPER_PACKAGE);

        //配置通用Mapper，详情请查阅官方文档
        Properties properties = new Properties();
        //通用定制版MyBatis Mapper插件接口
        properties.setProperty("mappers", Mapper.class.getName());
        //空字符串不插入和修改 注意true时已有数据置空也不再修改 insert、update是否判断字符串类型!='' 即 test="str != null"表达式内是否追加 and str != ''
        properties.setProperty("notEmpty", "false");
        //生成主键的方式
        properties.setProperty("IDENTITY", "MYSQL");
        //主键UUID回写方法执行顺序 默认AFTER  可选值为BEFORE|AFTER 。 BEFORE导致自增主键返回为null或0
        //可配合 @GeneratedValue(strategy = GenerationType.IDENTITY, generator = GenIdKey.UUID_SHORT ) 使用自定义分布式主键
        //properties.setProperty("ORDER", "BEFORE");
        //配置是否将枚举类型当成基本类型 默认情况下，枚举不会当作表中的字段
        //properties.setProperty("enumAsSimpleType", "true");
        mapperScannerConfigurer.setProperties(properties);

        return mapperScannerConfigurer;
    }
}


