package com.onsiteservice.common.config;


import lombok.extern.slf4j.Slf4j;
import org.springframework.amqp.core.AcknowledgeMode;
import org.springframework.amqp.rabbit.annotation.EnableRabbit;
import org.springframework.amqp.rabbit.config.SimpleRabbitListenerContainerFactory;
import org.springframework.amqp.rabbit.connection.ConnectionFactory;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.amqp.support.converter.Jackson2JsonMessageConverter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

/**
 * @author 潘维吉
 * @date 2020-01-12
 * RabbitMQ消息队列配置
 * AMQP，即Advanced Message Queuing Protocol，高级消息队列协议，
 * 是应用层协议的一个开放标准，为面向消息的中间件设计。
 * 消息中间件主要用于组件之间的解耦，消息的发送者无需知道消息使用者的存在，反之亦然。
 * AMQP的主要特征是面向消息、队列、路由（包括点对点和发布/订阅）、可靠性、安全。
 * 主要的功能是用来实现应用服务的异步与解耦，同时也能起到削峰填谷、消息分发的作用
 */
@ConditionalOnProperty(prefix = "project.rabbitmq", name = {"enabled"}, matchIfMissing = false)
@Configuration
@EnableRabbit
@Slf4j
public class RabbitMQConfig {

    /** 配置RabbitMQ消息服务 */
    @Autowired
    public ConnectionFactory connectionFactory;

    /**
     * ConnectionFactory类似于数据连接等
     *
     * @return SimpleRabbitListenerContainerFactory
     */
    @Bean
    public SimpleRabbitListenerContainerFactory rabbitListenerContainerFactory() {
        SimpleRabbitListenerContainerFactory factory = new SimpleRabbitListenerContainerFactory();
        factory.setConnectionFactory(connectionFactory);
        // 消息转换器, 如将Java类转换JSON类型发送至Broker, 从Broker处获取JSON消息转换为Java类型
        factory.setMessageConverter(new Jackson2JsonMessageConverter());
        // 注意 开启ACK  NONE自动确认 MANUAL手动确认 AUTO根据情况确认
        factory.setAcknowledgeMode(AcknowledgeMode.AUTO);
        factory.setConcurrentConsumers(1);
        factory.setMaxConcurrentConsumers(50);
        return factory;
    }

    /**
     * RabbitTemplate，用来发送消息
     *
     * @return RabbitTemplate
     */
    @Bean
    public RabbitTemplate rabbitTemplate() {
        RabbitTemplate template = new RabbitTemplate(connectionFactory);
        template.setMessageConverter(new Jackson2JsonMessageConverter());
        return template;
    }

    /**
     * 手动定义消息队列
     * 也可以根据@RabbitListener(queuesToDeclare = @Queue()) 自动创建队列
     */
/*    @Bean
    public Queue messageQueue() {
        return new Queue(SysConstants.Queue.MESSAGE, true);
    }*/

    /**
     * 定义消息队列
     *
     * @return Queue
     */
/*    @Bean
    public Queue queue() {
        // 创建字符串消息队列 boolean值代表是否持久化消息
        //durable 是否消息持久化
        //autoDelete 是否自动删除，即服务端或者客服端下线后 交换机自动删除
        return new Queue(QUEUE_NAME, true);
    }*/

    /**
     * Direct交换机
     *
     * 最为简单 先策略匹配到对应绑定的队列后 才会被投送到该队列  交换机跟队列必须是精确的对应关系
     *
     * @return DirectExchange
     */
/*    @Bean
    public DirectExchange directExchange() {
        return new DirectExchange(EXCHANGE_NAME, true, false);
    }*/

    /**
     * 创建 topic 类型的交换器
     * 交换机(Exchange) 描述：接收消息并且转发到绑定的队列，交换机不存储消息
     *
     * @return TopicExchange
     */
/*    @Bean
    public TopicExchange topicExchange() {
        return new TopicExchange(EXCHANGE_NAME, true, false);
    }*/

    /**
     * 建立交换机与队列的绑定关系
     *
     * @return
     */
/*    @Bean
    public Binding binding() {
        return BindingBuilder.bind(queue()).to(topicExchange()).with(ROUTING_KEY);
    }*/

    /**
     * 需要将ACK修改为手动确认，避免消息在处理过程中发生异常造成被误认为已经成功消费的假象
     * 使用@RabbitListener注释代替
     * @return SimpleMessageListenerContainer
     */
/*    @Bean
    public SimpleMessageListenerContainer messageListenerContainer() {
        SimpleMessageListenerContainer container = new SimpleMessageListenerContainer();
        container.setConnectionFactory(connectionFactory);
        // 监听的队列
        container.setQueues(queue());
        container.setExposeListenerChannel(true);
        container.setConcurrentConsumers(1);
        container.setMaxConcurrentConsumers(10);
        //开启ACK  NONE自动确认 MANUAL手动确认 AUTO根据情况确认
        container.setAcknowledgeMode(AcknowledgeMode.MANUAL);
        // 消息监听处理
        container.setMessageListener((ChannelAwareMessageListener) (message, channel) -> {
            byte[] body = message.getBody();
            log.info("RabbitMQ消费端接收到消息 : " + new String(body));
            if (message.getMessageProperties().getHeaders().get("error") == null) {
                // 确认消息成功消费
                channel.basicAck(message.getMessageProperties().getDeliveryTag(), false);
                log.info("RabbitMQ消息已经确认");
            } else {
                channel.basicReject(message.getMessageProperties().getDeliveryTag(), false);
                log.info("RabbitMQ消息拒绝");
            }
        });
        return container;
    }*/
}
