package com.onsiteservice.common.annotation.dict;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.onsiteservice.common.runner.InitDataRunner;
import com.onsiteservice.core.exception.ServiceException;
import com.onsiteservice.core.result.Result;
import com.onsiteservice.dao.common.page.PageInfoVO;
import com.onsiteservice.util.ArrayUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.stereotype.Component;

import java.lang.reflect.Method;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author 潘维吉
 * @date 2020/3/13 16:59
 * @email 406798106@qq.com
 * @description 字典code value响应数据切面 实现字典值的自动翻译
 */
@ConditionalOnProperty(prefix = "project.dict", name = {"enabled"}, matchIfMissing = true)
@Slf4j
@Aspect
@Component
public class DictAspect {

    @Pointcut("@annotation(com.onsiteservice.common.annotation.dict.Dict)")
    public void dict() {
    }

    @Around("dict()")
    public Object around(ProceedingJoinPoint joinPoint) throws Throwable {
        Object result = joinPoint.proceed();
        if (result instanceof Result) {
            Dict dict = getDict(joinPoint);
            Object data = ((Result) result).getData();

            if (data instanceof List) {  // 普通列表
                List newResult = new ArrayList();
                ((List) data).stream().forEach(item ->
                        newResult.add(handleData(dict, ((JSONObject) JSON.toJSON(item))))
                );
                return ((Result) result).setData(newResult);
            } else if (data instanceof PageInfoVO) { // 分页列表
                List newResult = new ArrayList();
                ((PageInfoVO) data).getList().stream().forEach(item ->
                        newResult.add(handleData(dict, ((JSONObject) JSON.toJSON(item))))
                );
                ((PageInfoVO) data).setList(newResult);
                return ((Result) result).setData(data);
            } else { // 单条数据
                JSONObject jsonObject = handleData(dict, (JSONObject) JSON.toJSON(data));
                return ((Result) result).setData(jsonObject);
            }
        }
        return result;
    }

    /**
     * 非注解方式翻译字典数据
     *
     * @param typeCode 主表编码
     * @param dictCode 字典表编码
     */
    public static String translateDictData(String typeCode, String dictCode) {
        List<Map> list = InitDataRunner.dictData.get(typeCode).stream()
                .filter(item -> item.get("code").equals(dictCode))
                .collect(Collectors.toList());
        // 字典值
        return ArrayUtils.isEmpty(list) ? null : list.get(0).get("value").toString();
    }

    /**
     * 字典表翻译值
     *
     * @param dict
     * @param jsonObject
     */
    private JSONObject handleData(Dict dict, JSONObject jsonObject) {
        List codeList = Arrays.asList(dict.code());
        List nameList = Arrays.asList(dict.name());
        List<Map> dictList = new ArrayList(2);
        for (int i = 0; i < codeList.size(); i++) {
            dictList.add(Map.of("code", codeList.get(i), "value", nameList.get(i)));
        }
        dictList.forEach(item -> {
            String codeName = item.get("value").toString();
            Object nameValue = jsonObject.get(item.get("value"));
            if (Objects.isNull(nameValue)) {
                return;
            }
            if (nameValue instanceof Boolean) {
                nameValue = BooleanUtils.toInteger((Boolean) nameValue);
                jsonObject.put(codeName, nameValue);
            } else if (nameValue instanceof Integer) {
                nameValue = nameValue;
            } else if (nameValue instanceof String) {
                nameValue = nameValue;
            } else {
                throw new ServiceException(String.format("字典值 %s:%s 类型错误, String或Integer或Boolean类型", codeName, nameValue));
            }
            try {
                Object finalNameValue = nameValue + "";
                List<Map> list = InitDataRunner.dictData.get(item.get("code")).stream()
                        .filter(items -> items.get("code").equals(finalNameValue))
                        .collect(Collectors.toList());
                // 平级结构 简化开发
                jsonObject.put(codeName + "Value", ArrayUtils.isEmpty(list) ? null : list.get(0).get("value"));
/*            jsonObject.put(codeName, Map.of("code", jsonObject.get(codeName),
                    "value", ArrayUtils.isEmpty(list) ? null : list.get(0).get("value")));*/
            } catch (NullPointerException e) {
                throw new ServiceException(String.format("字典数据%s不存在", codeName));
            } catch (Exception e) {
                throw new ServiceException("字典数据翻译异常");
            }
        });

        return jsonObject;
    }

    /**
     * 获取注解对象
     *
     * @param joinPoint 对象
     */
    private Dict getDict(final JoinPoint joinPoint) {
        Method[] methods = joinPoint.getTarget().getClass().getDeclaredMethods();
        String name = joinPoint.getSignature().getName();
        if (!StringUtils.isEmpty(name)) {
            for (Method method : methods) {
                Dict annotation = method.getAnnotation(Dict.class);
                if (!Objects.isNull(annotation) && name.equals(method.getName())) {
                    return annotation;
                }
            }
        }
        return null;
    }

}
