package com.onsiteservice.admin.security.authentication;

import com.alibaba.fastjson.JSON;
import com.onsiteservice.admin.security.pojo.AccountCredentials;
import com.onsiteservice.admin.security.pojo.CustomUserDetails;
import com.onsiteservice.admin.security.pojo.TokenTimeProperties;
import com.onsiteservice.admin.security.pojo.TokenVO;
import com.onsiteservice.core.config.MvcConfig;
import com.onsiteservice.core.result.Result;
import com.onsiteservice.core.security.jwt.JwtManager;
import lombok.extern.slf4j.Slf4j;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.authentication.InternalAuthenticationServiceException;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.web.authentication.AbstractAuthenticationProcessingFilter;
import org.springframework.security.web.util.matcher.AntPathRequestMatcher;

import javax.servlet.FilterChain;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.Collections;
import java.util.Map;
import java.util.Objects;

import static com.onsiteservice.core.result.ResultGenerator.failCustom;
import static com.onsiteservice.core.result.ResultGenerator.success;
import static org.springframework.http.HttpStatus.UNAUTHORIZED;

/**
 * @author 潘维吉
 * @date 2020/3/6 10:03
 * @email 406798106@qq.com
 * @description 将JWT添加到我们的身份验证过程
 */
@Slf4j
public class JwtLoginFilter extends AbstractAuthenticationProcessingFilter {

    public JwtLoginFilter(String url, AuthenticationManager authManager) {
        super(new AntPathRequestMatcher(url));
        setAuthenticationManager(authManager);
    }

    /**
     * 首先过滤器会调用自身的attemptAuthentication方法，从request中取出authentication,
     * authentication是在SecurityContextPersistenceFilter过滤器中通过捕获用户提交的登录表单中的内容生成的一个Authentication接口实例
     * 拿到authentication对象后，过滤器会调用authenticate认证授权方法，并传入该对象
     */
    @Override
    public Authentication attemptAuthentication(HttpServletRequest request, HttpServletResponse response)
            throws AuthenticationException, IOException {
        log.info("1. JwtLoginFilter.attemptAuthentication");
        AccountCredentials account = JSON.parseObject(request.getInputStream(), AccountCredentials.class);
        request.setAttribute("rememberMe", account.getRememberMe());
        UsernamePasswordAuthenticationToken token = new UsernamePasswordAuthenticationToken(
                account.getUserName(), account.getPassword(), Collections.emptyList());
        return getAuthenticationManager().authenticate(token);
    }

    /**
     * 认证成功 响应数据
     */
    @Override
    protected void successfulAuthentication(HttpServletRequest request, HttpServletResponse response, FilterChain chain, Authentication auth) {
        log.info("3. JwtLoginFilter.successfulAuthentication");
        Object rmObj = request.getAttribute("rememberMe");
        boolean rememberMe = rmObj != null && (Boolean) rmObj;

        // Authentication 中包含 loadUserByUsername() 方法返回的自定义的用户数据信息
        String userName = auth.getName();
        Long userId = ((CustomUserDetails) auth.getPrincipal()).getUserId();

        // 生成token 返回给前端
        String token = JwtManager.createToken(Map.of(JwtManager.USER_ID, userId, JwtManager.USER_NAME, userName),
                rememberMe ? TokenTimeProperties.rememberExpiresTime : TokenTimeProperties.expiresTime);

        Result result = success(TokenVO.builder().token(token)
                .expiresTime(rememberMe ? TokenTimeProperties.rememberExpiresTime : TokenTimeProperties.expiresTime).build(), "登录成功");
        MvcConfig.responseResult(response, result);
    }

    /**
     * 认证失败 响应数据
     */
    @Override
    protected void unsuccessfulAuthentication(HttpServletRequest request, HttpServletResponse response, AuthenticationException exception) {
        log.info("3. JwtLoginFilter.unsuccessfulAuthentication");
        exception.printStackTrace();
        String msg = (exception instanceof InternalAuthenticationServiceException) ?
                exception.getMessage() : "用户名或密码错误";
        Result result = failCustom(UNAUTHORIZED.value(), msg);
        MvcConfig.responseResult(response, result);
    }

}
