package com.onsiteservice.miniapp.controller.address;

import com.onsiteservice.common.annotation.user.CurrentUserId;
import com.onsiteservice.core.result.Result;
import com.onsiteservice.dao.common.page.PageInfoVO;
import com.onsiteservice.dao.common.page.PageParams;
import com.onsiteservice.miniapp.controller.address.dto.PageServiceAddressDTO;
import com.onsiteservice.miniapp.controller.address.dto.SaveServiceAddressDTO;
import com.onsiteservice.miniapp.controller.address.dto.UpdateServiceAddressDTO;
import com.onsiteservice.miniapp.controller.address.vo.ServiceAddressVO;
import com.onsiteservice.miniapp.service.address.ServiceAddressService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.NonNull;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import springfox.documentation.annotations.ApiIgnore;

import javax.validation.constraints.Positive;
import java.util.List;

import static com.onsiteservice.core.result.ResultGenerator.fail;
import static com.onsiteservice.core.result.ResultGenerator.success;

/**
 * <P>服务地址模块</P>
 *
 * @author 李宁
 * @version v1.0
 * @since 2022/7/7 08:51
 */
@Api(tags = "服务地址模块")
@RestController
@RequestMapping("/address/")
@Validated
public class ServiceAddressController {

    @Autowired
    private ServiceAddressService serviceAddressService;


    @ApiOperation(value = "分页查询列表")
    @PostMapping("page")
    public Result<PageInfoVO<ServiceAddressVO>> getPage(@RequestBody @NonNull @Validated PageServiceAddressDTO dto, @CurrentUserId Long uid) {
        return serviceAddressService.getPage(dto, uid);
    }

    @ApiOperation(value = "根据id查询服务地址")
    @GetMapping("get/{id}")
    public Result<ServiceAddressVO> getDetails(@ApiParam(name = "id", value = "服务地址id") @PathVariable @Positive Long id, @CurrentUserId Long userId) {
        return success(serviceAddressService.selectById(id, userId));
    }

    @ApiOperation(value = "设置默认地址")
    @PostMapping("def-address/{id}")
    public Result defaultAddress(@ApiParam(name = "id", value = "服务地址id") @PathVariable @Positive Long id, @CurrentUserId Long userId) {
        return serviceAddressService.defaultAddress(id, userId) == 1 ? success() : fail("设置默认地址失败");
    }

    @ApiOperation(value = "新增服务地址")
    @PostMapping("save")
    public Result save(@RequestBody @NonNull @Validated SaveServiceAddressDTO dto, @CurrentUserId Long userId) {
        return serviceAddressService.save(dto, userId) == 1 ? success() : fail("新增失败");
    }

    @ApiOperation(value = "修改服务地址")
    @PutMapping("update")
    public Result update(@RequestBody @NonNull @Validated UpdateServiceAddressDTO dto, @CurrentUserId Long userId) {
        return serviceAddressService.update(dto, userId) == 1 ? success() : fail("修改失败");
    }

    @ApiOperation(value = "根据id删除服务地址")
    @DeleteMapping("delete/{id}")
    public Result deleteById(@ApiParam(name = "id", value = "服务地址id") @PathVariable @Positive Long id, @CurrentUserId Long userId) {
        return serviceAddressService.delete(id, userId) == 1 ? success("删除成功") : fail("删除失败");
    }

}
