package com.onsiteservice.common.redis;

import org.springframework.data.redis.core.*;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.io.Serializable;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;

/**
 * @author 潘维吉
 * @date 2018-05-28
 * Redis操作工具类 封装
 * 可使用redisTemplate.executePipelined()流水线方式提高读写性能
 * 可开启Redis NoSQL事务功能
 */
@Service
@SuppressWarnings("unchecked")
public class RedisUtils {

    @Resource
    private RedisTemplate redisTemplate;

    /**
     * 写入缓存
     */
    public boolean set(final String key, Object value) {
        boolean result = false;
        try {
            ValueOperations<Serializable, Object> operations = redisTemplate.opsForValue();
            operations.set(key, value);
            result = true;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return result;
    }

    /**
     * 读取缓存
     */
    public Object get(final String key) {
        Object result = null;
        ValueOperations<Serializable, Object> operations = redisTemplate.opsForValue();
        result = operations.get(key);
        return result;
    }

    /**
     * 写入缓存设置时效时间 支持list类型等
     */
    public boolean set(final String key, Object value, Long expireTime, TimeUnit timeUnit) {
        boolean result = false;
        try {
            ValueOperations<Serializable, Object> operations = redisTemplate.opsForValue();
            operations.set(key, value);
            redisTemplate.expire(key, expireTime, timeUnit);
            result = true;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return result;
    }

    /**
     * 删除对应的key
     *
     * @param key
     */
    public boolean remove(final String key) {
        if (exists(key)) {
            return redisTemplate.delete(key);
        } else {
            return false;
        }
    }

    /**
     * 批量删除对应的key
     */
    public void remove(final String... keys) {
        for (String key : keys) {
            remove(key);
        }
    }

    /**
     * 正则批量删除key
     */
    public void removePattern(final String pattern) {
        Set<Serializable> keys = redisTemplate.keys(pattern);
        if (keys.size() > 0) {
            redisTemplate.delete(keys);
        }
    }

    /**
     * 判断缓存中是否有对应的value
     *
     * @param key
     * @return
     */
    public boolean exists(final String key) {
        return redisTemplate.hasKey(key);
    }

    /**
     * 列表添加
     */
    public void addList(String key, Object value) {
        ListOperations<String, Object> list = redisTemplate.opsForList();
        list.rightPush(key, value);
    }

    /**
     * 列表全部添加
     */
    public void addAllList(String key, Object value) {
        remove(key);   // 全部添加List 先清空
        ListOperations<String, Object> list = redisTemplate.opsForList();
        list.rightPushAll(key, value);
    }

    /**
     * 列表获取
     */
    public <T> List<T> getList(String key, long length, long length1) {
        ListOperations<String, Object> list = redisTemplate.opsForList();
        return (List<T>) list.range(key, length, length1);
    }

    /**
     * 获取列表全部
     */
    public <T> List<T> getAllList(String key) {
        ListOperations<String, Object> list = redisTemplate.opsForList();
        return (List<T>) list.range(key, 0, list.size(key) - 1);
    }

    /**
     * Set集合添加
     */
    public void addSet(String key, Object value) {
        SetOperations<String, Object> set = redisTemplate.opsForSet();
        set.add(key, value);
    }

    /**
     * Set集合获取
     */
    public Set<Object> getSet(String key) {
        SetOperations<String, Object> set = redisTemplate.opsForSet();
        return set.members(key);
    }

    /**
     * 删除Set集合数据
     *
     * @param key    为Set集合的key
     * @param values 为Set集合的多条具体数据value值
     */
    public void removeSet(String key, Object... values) {
        SetOperations<String, Object> set = redisTemplate.opsForSet();
        set.remove(key, values);
    }

    /**
     * 有序集合添加
     */
    public void addZSet(String key, Object value, double resources) {
        ZSetOperations<String, Object> zSet = redisTemplate.opsForZSet();
        zSet.add(key, value, resources);
    }

    /**
     * 有序集合获取
     */
    public Set<Object> getZSet(String key, double resources, double resources1) {
        ZSetOperations<String, Object> zSet = redisTemplate.opsForZSet();
        return zSet.rangeByScore(key, resources, resources1);
    }

    /**
     * 哈希 添加
     */
    public void setHash(String key, Object hashKey, Object value) {
        HashOperations<String, Object, Object> hash = redisTemplate.opsForHash();
        hash.put(key, hashKey, value);
    }

    /**
     * 哈希获取数据
     */
    public Object getHash(String key, Object hashKey) {
        HashOperations<String, Object, Object> hash = redisTemplate.opsForHash();
        return hash.get(key, hashKey);
    }

    /**
     * 重置失效时间
     */
    public void expire(final String key, Long expireTime, TimeUnit timeUnit) {
        redisTemplate.expire(key, expireTime, timeUnit);
    }
}
