package com.onsiteservice.admin.service.order;

import com.github.pagehelper.PageHelper;
import com.onsiteservice.admin.controller.order.convert.OrderConvert;
import com.onsiteservice.admin.controller.order.dto.OrderPageDTO;
import com.onsiteservice.admin.controller.order.vo.OrderDetailVO;
import com.onsiteservice.admin.controller.order.vo.OrderPageVO;
import com.onsiteservice.admin.mapper.order.ServiceOrderBizMapper;
import com.onsiteservice.common.order.vo.ServiceOrderLogVO;
import com.onsiteservice.constant.constant.BizConstants;
import com.onsiteservice.constant.enums.BizCodeEnum;
import com.onsiteservice.core.exception.ServiceException;
import com.onsiteservice.dao.common.AbstractMapper;
import com.onsiteservice.dao.common.page.PageInfoVO;
import com.onsiteservice.dao.mapper.service.ServiceOrderImgMapper;
import com.onsiteservice.dao.mapper.service.ServiceOrderLogMapper;
import com.onsiteservice.dao.mapper.service.ServiceSubclassMapper;
import com.onsiteservice.entity.order.ServiceOrder;
import com.onsiteservice.entity.service.ServiceOrderImg;
import com.onsiteservice.entity.service.ServiceOrderLog;
import com.onsiteservice.util.AttrCopyUtils;
import lombok.AllArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import tk.mybatis.mapper.entity.Condition;

import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author 商天翔
 * @date 2022-07-14 15:11
 * @description ServiceOrderService服务类
 */
@Service
@Transactional(rollbackFor = Exception.class)
@AllArgsConstructor(onConstructor_ = {@Autowired})
public class AdminOrderService extends AbstractMapper<ServiceOrder> {

    private ServiceOrderBizMapper serviceOrderBizMapper;
    private OrderConvert orderConvert;
    private ServiceOrderImgMapper serviceOrderImgMapper;
    private ServiceOrderLogMapper serviceOrderLogMapper;
    private ServiceSubclassMapper serviceSubclassMapper;

    /**
     * 分页查询列表
     */
    public PageInfoVO<OrderPageVO> getPage(OrderPageDTO param) {
        PageHelper.startPage(param.getPage(), param.getSize());
        return new PageInfoVO<>(serviceOrderBizMapper.getPage(param));
    }

    /**
     * 获取订单详情
     */
    public OrderDetailVO getOrderDetail(Long id) {
        // 订单信息
        ServiceOrder order = selectByPrimaryKey(id);
        if (order == null) {
            throw new ServiceException(BizCodeEnum.SERVICE_ORDER_NOT_EXIST);
        }
        // 基础订单信息
        var serviceOrderVO = orderConvert.toVO(order);
        // 服务信息
        var subService = serviceSubclassMapper.selectByPrimaryKey(order.getSubclassId());
        serviceOrderVO.setServiceName(subService.getServiceName());

        // 需求图片
        Condition c1 = new Condition(ServiceOrderImg.class);
        c1.createCriteria().andEqualTo(BizConstants.OrderConstants.ORDER_ID, serviceOrderVO.getId());
        List<ServiceOrderImg> serviceOrderImgList = serviceOrderImgMapper.selectByCondition(c1);
        List<String> serviceOrderImgUrlList = serviceOrderImgList.parallelStream().map(ServiceOrderImg::getUrl).collect(Collectors.toList());
        serviceOrderVO.setDemandImgUrls(serviceOrderImgUrlList);

        // 流程信息
        Condition c2 = new Condition(ServiceOrderLog.class);
        c2.createCriteria().andEqualTo(BizConstants.OrderConstants.ORDER_ID, id);
        List<ServiceOrderLog> serviceOrderLogList = serviceOrderLogMapper.selectByCondition(c2);
        List<ServiceOrderLogVO> serviceOrderLogVOList = serviceOrderLogList.parallelStream()
                .map(e -> AttrCopyUtils.copy(e, new ServiceOrderLogVO()))
                .sorted(Comparator.comparing(ServiceOrderLogVO::getCreateTime).reversed()).collect(Collectors.toList());
        serviceOrderVO.setProcess(serviceOrderLogVOList);

        return serviceOrderVO;
    }
}
