package com.onsiteservice.miniapp.service.weixin.pay;

import com.onsiteservice.core.exception.ServiceException;
import com.onsiteservice.miniapp.service.weixin.pay.config.WXPayConstants;
import com.onsiteservice.miniapp.service.weixin.pay.config.WechatPayMini;
import com.onsiteservice.miniapp.service.weixin.pay.config.WechatPayMyConfigMini;
import com.onsiteservice.miniapp.service.weixin.pay.util.*;
import com.onsiteservice.util.IPUtils;
import com.onsiteservice.util.UUIDUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.http.HttpServletRequest;
import java.io.ByteArrayInputStream;
import java.math.BigDecimal;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

/**
 * @author: qiaobin
 * @description:微信网页扫码支付
 * @date: 2019/1/22
 */

public class WechatNativePay {
    private static Logger logger = LoggerFactory.getLogger(WechatNativePay.class);

    WechatPayMyConfigMini configMini;
    WechatPayMini wxpayMini;

    public WechatNativePay(String appID, String mchID, String key) {
        configMini = new WechatPayMyConfigMini(appID, mchID, key);
        wxpayMini = new WechatPayMini(configMini);
    }

    public Map<String, String> getPayParam(String body, String out_trade_no, BigDecimal total_fee, String notifyUrl, String openId, HttpServletRequest request) throws Exception {
        //微信支付
        HashMap<String, String> payParam = new HashMap<>();
        String timestamp = String.valueOf(System.currentTimeMillis() / 1000);
        String prepayId = getPrepayId(body, out_trade_no, total_fee, notifyUrl, openId, request);
        payParam.put("appId", configMini.getAppID());
        payParam.put("timeStamp", timestamp);
        payParam.put("nonceStr", UUIDGenerator.generate());
        payParam.put("package", "prepay_id=" + prepayId);
        payParam.put("signType", "MD5");
        String paySign = WXPayUtil.generateSignature(payParam, configMini.getKey());
        payParam.put("paySign", paySign);
        return payParam;
    }

    /**
     * 生成预支付交易单
     *
     * @param body         商品或支付单简要描述 32个字符
     * @param out_trade_no 商户系统内部的订单号,32个字符内
     * @param total_fee    支付金额单位为【分】
     * @param notifyUrl
     * @param request
     * @return
     * @throws Exception
     */
    public String getPrepayId(String body, String out_trade_no, BigDecimal total_fee, String notifyUrl, String openId, HttpServletRequest request) throws Exception {
        logger.info("===getPrepayId 参数=====" + body + "|" + out_trade_no + "|" + total_fee.toString() + "|" + notifyUrl + "|" + openId);
        String prepayId = "";
        total_fee = total_fee.multiply(new BigDecimal(100));//微信支付金额需要乘以100
        Map<String, String> data = new HashMap<String, String>();
        data.put("body", body);
        data.put("appid", configMini.getAppID());
        data.put("mch_id", configMini.getMchID());
        data.put("out_trade_no", out_trade_no);
        data.put("fee_type", "CNY");
        data.put("openid", openId);
        data.put("total_fee", String.valueOf(total_fee.intValue()));
        data.put("spbill_create_ip", IPUtils.getIpAddr(request));
        data.put("notify_url", notifyUrl);
        data.put("trade_type", "JSAPI");  // 此处指定为小程序支付
        data.put("nonce_str", UUIDGenerator.generate());
        String s = MapUtil.FormatBizQueryParaMap(data, false);
        String buildSign = s + "&key=" + configMini.getKey();
        String sign = MD5Util.MD5(buildSign).toUpperCase();
        data.put("sign", sign);
        Map<String, String> respXml = wxpayMini.unifiedOrder(data);
        logger.info("===获取预支付id,返回xml数据=====" + respXml);
        String returnCode = respXml.get("return_code");
        String resultCode = respXml.get("result_code");
        if ("SUCCESS".equals(resultCode) && "SUCCESS".equals(returnCode)) {
            prepayId = respXml.get("prepay_id");
        } else {
            throw new RuntimeException("生成预支付id时未成功.");
        }
        return prepayId;
    }

    /**
     * 生成退款交易单
     *
     * @param out_trade_no  商户系统内部的订单号,32个字符内
     * @param out_refund_no 商户系统内部的退款订单号,32个字符内
     * @param total_fee     支付金额单位为【分】
     * @return
     * @throws Exception
     */
    public String doRefundMini(String out_trade_no, String out_refund_no, BigDecimal total_fee, BigDecimal refund_fee, String refundNotifyUrl, byte[] certData) throws Exception {

        String appId = configMini.getAppID();
        String mchId = configMini.getMchID();
        String wechatKey = configMini.getKey();
        String resultCode = "";
        HashMap<String, String> prepayMap = new HashMap<>();
        prepayMap.put("appid", appId);
        prepayMap.put("nonce_str", UUID.randomUUID().toString().replaceAll("-", ""));
        prepayMap.put("mch_id", mchId);
        prepayMap.put("notify_url", refundNotifyUrl);
        prepayMap.put("out_trade_no", out_trade_no);
        prepayMap.put("out_refund_no", out_refund_no);
        total_fee = total_fee.multiply(new BigDecimal(100)); //重要，微信支付金额是分，前端传来是支付金额单位是元，必须扩大100倍
        refund_fee = refund_fee.multiply(new BigDecimal(100)); //重要，微信支付金额是分，前端传来是支付金额单位是元，必须扩大100倍
        prepayMap.put("total_fee", String.valueOf(total_fee.intValue()));
        prepayMap.put("refund_fee", String.valueOf(refund_fee.intValue()));
        String s = MapUtil.FormatBizQueryParaMap(prepayMap, false);
        String buildSign = s + "&key=" + wechatKey;
        String sign = MD5Util.MD5(buildSign).toUpperCase();
        prepayMap.put("sign", sign);

        logger.info("===发起退款操作,参数=====appId" + appId);
        logger.info("===发起退款操作,参数=====nonce_str" + prepayMap.get("nonce_str"));
        logger.info("===发起退款操作,参数=====mch_id" + prepayMap.get("mch_id"));
        logger.info("===发起退款操作,参数=====notify_url" + prepayMap.get("notify_url"));
        logger.info("===发起退款操作,参数=====out_trade_no" + prepayMap.get("out_trade_no"));
        logger.info("===发起退款操作,参数=====out_refund_no" + prepayMap.get("out_refund_no"));
        logger.info("===发起退款操作,参数=====total_fee" + prepayMap.get("total_fee"));
        logger.info("===发起退款操作,参数=====refund_fee" + prepayMap.get("refund_fee"));
        logger.info("===发起退款操作,参数=====sign" + prepayMap.get("sign"));
        String sendXml = MapUtil.ArrayToXml(prepayMap);
        try {
            String returnXml = ClientCustomSSL.doRefund(new ByteArrayInputStream(certData), WXPayConstants.WXREFUND_URL, sendXml, mchId);
            logger.info("===发起退款操作,返回xml数据=====" + returnXml);
            String returnCode = ParseXmlUtil.parseXml2KeyValue(returnXml, "return_code");
            resultCode = ParseXmlUtil.parseXml2KeyValue(returnXml, "result_code");
            if (!"SUCCESS".equals(resultCode) || !"SUCCESS".equals(returnCode)) {
                throw new ServiceException("生成退款交易单时未成功.");
            }
        } catch (Throwable e) {
            e.printStackTrace();
            logger.info("=====发起退款异常====" + e.getMessage());
            throw new ServiceException(e.getMessage());
        }
        return resultCode;
    }
}
