package com.onsiteservice.miniapp.service.user;

import com.alibaba.fastjson.JSON;
import com.onsiteservice.common.runner.InitDataRunner;
import com.onsiteservice.common.service.CommonSmsService;
import com.onsiteservice.constant.constant.Constants;
import com.onsiteservice.constant.constant.DictConstants;
import com.onsiteservice.constant.constant.RedisKeyConstants;
import com.onsiteservice.constant.enums.BizCodeEnum;
import com.onsiteservice.constant.enums.SendCodeEnum;
import com.onsiteservice.constant.enums.ServiceOrderStatusEnum;
import com.onsiteservice.constant.enums.ServiceUserTypeEnum;
import com.onsiteservice.core.exception.ServiceException;
import com.onsiteservice.core.result.Result;
import com.onsiteservice.core.result.ResultCodeEnum;
import com.onsiteservice.core.result.ResultGenerator;
import com.onsiteservice.core.security.jwt.JwtManager;
import com.onsiteservice.core.security.jwt.JwtPathProperties;
import com.onsiteservice.dao.common.AbstractMapper;
import com.onsiteservice.dao.mapper.service.ServiceOrderMapper;
import com.onsiteservice.dao.mapper.service.ServiceValuatorAssignMapper;
import com.onsiteservice.dao.mapper.service.ServiceWorkerAssignMapper;
import com.onsiteservice.dao.mapper.user.UserMapper;
import com.onsiteservice.entity.order.ServiceOrder;
import com.onsiteservice.entity.service.ServiceValuatorAssign;
import com.onsiteservice.entity.user.User;
import com.onsiteservice.miniapp.controller.user.convert.UserConvert;
import com.onsiteservice.miniapp.controller.user.dto.BindPhoneDTO;
import com.onsiteservice.miniapp.controller.user.dto.SendCodeDTO;
import com.onsiteservice.miniapp.controller.user.vo.UserInfoVO;
import com.onsiteservice.miniapp.controller.user.vo.UserOrderInfoVO;
import com.onsiteservice.miniapp.mapper.order.ServiceOrderBizMapper;
import com.onsiteservice.util.RandomUtils;
import io.jsonwebtoken.Claims;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import tk.mybatis.mapper.entity.Condition;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

@Service
@Slf4j
@Transactional(rollbackFor = Exception.class)
@AllArgsConstructor(onConstructor_ = {@Autowired})
public class UserService extends AbstractMapper<User> {

    private StringRedisTemplate redisTemplate;
    private UserMapper userMapper;
    private UserConvert userConvert;
    private ServiceOrderBizMapper serviceOrderBizMapper;
    private ServiceOrderMapper serviceOrderMapper;
    private ServiceValuatorAssignMapper serviceValuatorAssignMapper;
    private CommonSmsService commonSmsService;


    public Result<String> sendCode(SendCodeDTO dto) {
        log.info("user sendCode dto: {}", dto);

        String cacheKey = String.format(RedisKeyConstants.CHECK_CODE_KEY, SendCodeEnum.REBIND_PHONE.name(), dto.getTo());
        String cacheValue = redisTemplate.opsForValue().get(cacheKey);
        // 重复发送逻辑
        if (StringUtils.isNotBlank(cacheValue)) {
            long ttl = Long.parseLong(cacheValue.split("_")[1]);
            if ((System.currentTimeMillis() - ttl) < 60 * 1000) {
                throw new ServiceException(BizCodeEnum.CODE_LIMITED);
            }
        }

        String code = RandomUtils.generateNum(6);
        String value = code + "_" + System.currentTimeMillis();
        log.info("user sendCode phone: {}, code: {}", dto.getTo(), code);

        // 置入
        redisTemplate.opsForValue().set(cacheKey, value, RedisKeyConstants.CODE_EXPIRED, TimeUnit.MILLISECONDS);

        commonSmsService.sendChangePhone(code, dto.getTo());

        return ResultGenerator.success();
    }


    public Result<String> rebindPhone(BindPhoneDTO dto, Long userId) {
        log.info("user rebindPhone dto: {}, userId: {}", dto, userId);

        boolean f = checkCode(SendCodeEnum.REBIND_PHONE, dto.getPhone(), dto.getCode());
        if (!f) {
            throw new ServiceException(BizCodeEnum.CODE_ERROR);
        }

        User user = User.builder().id(userId).phone(dto.getPhone()).build();
        return userMapper.updateByPrimaryKeySelective(user) == 1 ? ResultGenerator.success() : ResultGenerator.fail("换绑失败");
    }


    /**
     * 验证 code
     */
    public boolean checkCode(SendCodeEnum sendCodeEnum, String to, String code) {
        String cacheKey = String.format(RedisKeyConstants.CHECK_CODE_KEY, sendCodeEnum.name(), to);
        String cacheValue = redisTemplate.opsForValue().get(cacheKey);
        if (StringUtils.isNotBlank(cacheValue)) {
            String cacheCode = cacheValue.split("_")[0];
            if (cacheCode.equalsIgnoreCase(code)) {
                // 删除验证码
                redisTemplate.delete(cacheKey);
                return true;
            }
        }

        return false;
    }

    /**
     * 获取用户信息
     */
    public UserInfoVO getInfo(Long userId) {
        var user = userMapper.selectByPrimaryKey(userId);
        var vo = userConvert.toUserVO(user);
        vo.setSex(InitDataRunner.dictData.get(DictConstants.SEX).get(vo.getSexValue()).get("value").toString());

        // 待处理数量
        if (Objects.equals(user.getRoleType(), ServiceUserTypeEnum.ADMIN.getId())) {
            Condition condition = new Condition(ServiceOrder.class);
            condition.createCriteria().andEqualTo("orderStatus", ServiceOrderStatusEnum.RESERVE.getStatus());
            int num = serviceOrderMapper.selectCountByCondition(condition);
            vo.setTodo(num);
        }
        if (Objects.equals(user.getRoleType(), ServiceUserTypeEnum.VALUATOR.getId())) {
            Condition condition = new Condition(ServiceValuatorAssign.class);
            condition.createCriteria().andEqualTo("valuatorId", userId)
                    .andGreaterThan("assignTime", new Date());
            int num = serviceValuatorAssignMapper.selectCountByCondition(condition);
            vo.setTodo(num);
        }
        // 获取订单状况
        var numMap = serviceOrderBizMapper.getMyOrderNum(userId);
        vo.setOrderInfo(List.of(
                new UserOrderInfoVO("待支付", numMap.get("pay"), 1),
                new UserOrderInfoVO("待受理", numMap.get("handle"), 3),
                new UserOrderInfoVO("待派单", numMap.get("dispatch"), 4),
                new UserOrderInfoVO("已完成", numMap.get("finished"), 6)
        ));
        return vo;
    }

    public Integer getRoleType(HttpServletRequest request) {
        // 获取token
        String token = request.getHeader(JwtManager.AUTHORIZATION_HEADER);
        if (StringUtils.isEmpty(token)) {
            return ServiceUserTypeEnum.USER.getId();
        }
        //
        Claims claims = JwtManager.parseToken(token.replaceAll(JwtManager.BEARER, ""));
        if (claims != null) {
            // 如果token验证成功，将token对应的用户id
            Object userId = JSON.parseObject(claims.getSubject()).get(JwtManager.USER_ID);
            if (userId != null) {
                var user = userMapper.selectByPrimaryKey(userId);
                if (user != null) {
                    return user.getRoleType();
                }
            }
        }
        return ServiceUserTypeEnum.USER.getId();

    }
}
