package com.onsiteservice.util.excel;

import cn.afterturn.easypoi.excel.ExcelExportUtil;
import cn.afterturn.easypoi.excel.ExcelImportUtil;
import cn.afterturn.easypoi.excel.entity.ExportParams;
import cn.afterturn.easypoi.excel.entity.ImportParams;
import cn.afterturn.easypoi.excel.entity.TemplateExportParams;
import cn.afterturn.easypoi.excel.entity.enmus.ExcelType;
import cn.afterturn.easypoi.excel.entity.params.ExcelExportEntity;
import com.aliyun.oss.ServiceException;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.ss.usermodel.Workbook;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.File;
import java.io.IOException;
import java.net.URLEncoder;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;

/**
 * @author 潘维吉
 * @date 2019/10/29 16:45
 * @email 406798106@qq.com
 * @description Excel文件导入导出工具类
 */
public class ExcelUtils {

    /**
     * 根据文件流 导入excel
     *
     * @param file
     * @param titleRows
     * @param headerRows
     * @param pojoClass
     * @param <T>
     * @return
     */
    public static <T> List<T> importExcel(MultipartFile file, Integer titleRows, Integer headerRows,
                                          Class<T> pojoClass) {
        if (file == null) {
            return null;
        }
        ImportParams params = new ImportParams();
        params.setTitleRows(titleRows);
        params.setHeadRows(headerRows);
        List<T> list = null;
        try {
            list = ExcelImportUtil.importExcel(file.getInputStream(), pojoClass, params);
        } catch (NoSuchElementException e) {
            throw new RuntimeException("excel文件不能为空");
        } catch (Exception e) {
            throw new RuntimeException(e.getMessage());
        }
        return list;
    }

    /**
     * 根据文件路径 导入excel
     *
     * @param filePath
     * @param titleRows
     * @param headerRows
     * @param pojoClass
     * @param <T>
     * @return
     */
    public static <T> List<T> importExcel(String filePath, Integer titleRows, Integer headerRows, Class<T> pojoClass) {
        if (StringUtils.isBlank(filePath)) {
            return null;
        }
        ImportParams params = new ImportParams();
        params.setTitleRows(titleRows);
        params.setHeadRows(headerRows);
        List<T> list = null;
        try {
            list = ExcelImportUtil.importExcel(new File(filePath), pojoClass, params);
        } catch (NoSuchElementException e) {
            throw new RuntimeException("模板不能为空");
        } catch (Exception e) {
            throw new RuntimeException(e.getMessage());
        }
        return list;
    }

    /**
     * 导出excel
     *
     * @param list
     * @param title
     * @param sheetName
     * @param pojoClass
     * @param fileName
     * @param response
     */
    public static void exportExcel(List<?> list, String title, String sheetName, Class<?> pojoClass, String fileName,
                                   HttpServletResponse response) {
        ExcelExportUtil.USE_SXSSF_LIMIT = 10000;
        defaultExport(list, pojoClass, fileName, response, new ExportParams(title, sheetName));
    }

    public static void exportExcel(List<?> list, String title, String sheetName, Class<?> pojoClass, String fileName,
                                   HttpServletRequest request, HttpServletResponse response) {
        ExcelExportUtil.USE_SXSSF_LIMIT = 10000;
        ExportParams param = new ExportParams(title, sheetName, ExcelType.XSSF);
        param.setStyle(ExcelExportStyleImpl.class);
        defaultExport(list, pojoClass, fileName, request, response, param);
    }

    /**
     * @param mapList
     * @param entityList 列设置，该list的顺序决定Excel列顺序
     * @param title
     * @param sheetName
     * @param fileName
     * @param request
     * @param response
     */
    public static void mapExportExcel(List<Map<String, Object>> mapList, List<ExcelExportEntity> entityList,
                                      String title, String sheetName, String fileName,
                                      HttpServletRequest request, HttpServletResponse response) {
        ExcelExportUtil.USE_SXSSF_LIMIT = 10000;
        mapExport(mapList, entityList, fileName, request, response, new ExportParams(title, sheetName, ExcelType.XSSF));
    }

    public static void mapExportExcel(List<Map<String, Object>> mapList, List<ExcelExportEntity> entityList,
                                       String title, String sheetName, String fileName, HttpServletResponse response) {
        ExcelExportUtil.USE_SXSSF_LIMIT = 10000;
        Workbook workbook = ExcelExportUtil.exportExcel(new ExportParams(title, sheetName), entityList, mapList);
        if (workbook != null) {
            downLoadExcel(fileName, response, workbook);
        }
    }

    /**
     * 导出excel  是否创建文件头
     *
     * @param list
     * @param title
     * @param sheetName
     * @param pojoClass
     * @param fileName
     * @param isCreateHeader
     * @param response
     */
    public static void exportExcel(List<?> list, String title, String sheetName, Class<?> pojoClass, String fileName,
                                   boolean isCreateHeader, HttpServletResponse response) {
        ExcelExportUtil.USE_SXSSF_LIMIT = 10000;
        ExportParams exportParams = new ExportParams(title, sheetName);
        exportParams.setCreateHeadRows(isCreateHeader);
        defaultExport(list, pojoClass, fileName, response, exportParams);
    }

    public static void exportExcel(List<Map<String, Object>> list, String fileName, HttpServletResponse response) {
        defaultExport(list, fileName, response);
    }

    /**
     * 根据模板导出excel 不支持某些版本的WPS生成的excel文件
     * 参考 http://doc.wupaas.com/docs/easypoi/easypoi-1c10lfhut694k
     * @param templatePath 模板文件路径
     * @param data         导出的数据
     */
    public static void exportWithTemplate(String templatePath, Map<String, Object> data, String fileName, HttpServletRequest request, HttpServletResponse response) {
        TemplateExportParams params = new TemplateExportParams(templatePath);
        Workbook workbook = ExcelExportUtil.exportExcel(params, data);
        if (workbook != null) {
            downLoadExcel(fileName, request, response, workbook);
        }
    }

    private static void defaultExport(List<?> list, Class<?> pojoClass, String fileName, HttpServletResponse response,
                                      ExportParams exportParams) {
        Workbook workbook = ExcelExportUtil.exportExcel(exportParams, pojoClass, list);
        if (workbook != null) {
            downLoadExcel(fileName, response, workbook);
        }
    }

    private static void defaultExport(List<?> list, Class<?> pojoClass, String fileName, HttpServletRequest request, HttpServletResponse response,
                                      ExportParams exportParams) {
        Workbook workbook = ExcelExportUtil.exportExcel(exportParams, pojoClass, list);
        if (workbook != null) {
            downLoadExcel(fileName, request, response, workbook);
        }
    }

    private static void defaultExport(List<Map<String, Object>> list, String fileName, HttpServletResponse response) {
        Workbook workbook = ExcelExportUtil.exportExcel(list, ExcelType.HSSF);
        if (workbook != null) {
            downLoadExcel(fileName, response, workbook);
        }
    }

    private static void mapExport(List<Map<String, Object>> mapList, List<ExcelExportEntity> entityList, String fileName, HttpServletRequest request, HttpServletResponse response,
                                  ExportParams exportParams) {
        Workbook workbook = ExcelExportUtil.exportExcel(exportParams, entityList, mapList);
        if (workbook != null) {
            downLoadExcel(fileName, request, response, workbook);
        }
    }

    /**
     * 下载excel
     *
     * @param fileName
     * @param response
     * @param workbook
     */
    private static void downLoadExcel(String fileName, HttpServletResponse response, Workbook workbook) {
        try {
            response.setCharacterEncoding("UTF-8");
            response.setHeader("content-Type", "application/vnd.ms-excel");
            response.setHeader("Content-Disposition", "attachment;filename=" + URLEncoder.encode(fileName, "UTF-8"));
            workbook.write(response.getOutputStream());
        } catch (IOException e) {
            throw new RuntimeException(e.getMessage());
        }
    }

    public static void downLoadExcel(String fileName, HttpServletRequest request, HttpServletResponse response, Workbook workbook) {
        try {
            response.reset();
            String origin = request.getHeader("Origin");
            response.setHeader("Access-Control-Allow-Origin", origin);
            response.setHeader("Access-Control-Allow-Credentials", "true");
            response.setContentType("application/octet-stream;charset=utf-8");
            response.setHeader("Content-Disposition", "attachment;filename=" + URLEncoder.encode(fileName, "UTF-8"));
            Cookie fileDownload = new Cookie("fileDownload", "true");
            fileDownload.setPath("/");
            response.addCookie(fileDownload);
            workbook.write(response.getOutputStream());
        } catch (IOException e) {
            throw new ServiceException(e.getMessage());
        }
    }

}
