package com.onsiteservice.miniapp.service.worker;

import com.google.common.collect.Lists;
import com.onsiteservice.dao.common.AbstractMapper;
import com.onsiteservice.dao.mapper.service.ServiceWorkerAssignMapper;
import com.onsiteservice.entity.service.ServiceWorker;
import com.onsiteservice.entity.service.ServiceWorkerAssign;
import com.onsiteservice.miniapp.controller.worker.vo.ServiceWorkerVO;
import com.onsiteservice.miniapp.mapper.service.ServiceWorkerBizMapper;
import com.onsiteservice.util.AttrCopyUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author 潘维吉
 * @date 2022-07-11 16:07
 * @description ServiceWorkerService服务类
 */
@Service
@Slf4j
@Transactional(rollbackFor = Exception.class)
public class ServiceWorkerService extends AbstractMapper<ServiceWorker> {

    @Resource
    private ServiceWorkerAssignMapper serviceWorkerAssignMapper;
    @Resource
    private ServiceWorkerBizMapper serviceWorkerBizMapper;


    public List<ServiceWorkerVO> all() {
        List<ServiceWorker> serviceWorkerList = this.selectAll();
        return handle(serviceWorkerList);
    }


    public List<ServiceWorkerVO> selectByName(String name) {
        log.info("worker selectByName name: {}", name);

        List<ServiceWorker> serviceWorkerList = serviceWorkerBizMapper.selectByNameLike(name);
        return handle(serviceWorkerList);
    }


    private List<ServiceWorkerVO> handle(List<ServiceWorker> serviceWorkerList) {
        if (CollectionUtils.isEmpty(serviceWorkerList)) {
            return Lists.newArrayList();
        }

        // 估价员id列表
        List<Long> workerIds = serviceWorkerList.parallelStream().map(ServiceWorker::getId).collect(Collectors.toList());
        // 已被指派的时间
        List<ServiceWorkerAssign> serviceWorkerAssignList = serviceWorkerAssignMapper.selectByDeletedAndIdListAndAssignTime(workerIds);
        Map<Long, List<ServiceWorkerAssign>> serviceWorkerAssignMap =
                serviceWorkerAssignList.parallelStream()
                        .collect(Collectors.groupingBy(ServiceWorkerAssign::getWorkerId));

        return serviceWorkerList.parallelStream().map(e1 -> {
            ServiceWorkerVO serviceWorkerVO = AttrCopyUtils.copy(e1, new ServiceWorkerVO());
            // 维修工已经被指派的时间
            List<ServiceWorkerAssign> assigns = serviceWorkerAssignMap.getOrDefault(e1.getId(), Lists.newArrayList());
            List<Date> assignTimes = assigns.parallelStream().map(ServiceWorkerAssign::getAssignTime).collect(Collectors.toList());

            serviceWorkerVO.setAssignTime(assignTimes);

            return serviceWorkerVO;
        }).collect(Collectors.toList());
    }


}
