package com.onsiteservice.miniapp.service.user;

import com.onsiteservice.constant.constant.RedisKeyConstants;
import com.onsiteservice.constant.enums.BizCodeEnum;
import com.onsiteservice.constant.enums.SendCodeEnum;
import com.onsiteservice.core.exception.ServiceException;
import com.onsiteservice.core.result.Result;
import com.onsiteservice.core.result.ResultGenerator;
import com.onsiteservice.dao.common.AbstractMapper;
import com.onsiteservice.dao.mapper.user.UserMapper;
import com.onsiteservice.entity.user.User;
import com.onsiteservice.miniapp.controller.user.dto.BindPhoneDTO;
import com.onsiteservice.miniapp.controller.user.dto.SendCodeDTO;
import com.onsiteservice.util.RandomUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.concurrent.TimeUnit;

@Service
@Slf4j
@Transactional(rollbackFor = Exception.class)
public class UserService extends AbstractMapper<User> {


    @Autowired
    private StringRedisTemplate redisTemplate;

    @Resource
    private UserMapper userMapper;


    public Result<String> sendCode(SendCodeDTO dto) {
        log.info("user sendCode dto: {}", dto);

        String cacheKey = String.format(RedisKeyConstants.CHECK_CODE_KEY, SendCodeEnum.REBIND_PHONE.name(), dto.getTo());
        String cacheValue = redisTemplate.opsForValue().get(cacheKey);
        // 重复发送逻辑
        if (StringUtils.isNotBlank(cacheValue)) {
            long ttl = Long.parseLong(cacheValue.split("_")[1]);
            if ((System.currentTimeMillis() - ttl) < 60 * 1000) {
                throw new ServiceException(BizCodeEnum.CODE_LIMITED);
            }
        }

        String code = RandomUtils.generateNum(6);
        String value = code + "_" + System.currentTimeMillis();
        log.info("user sendCode phone: {}, code: {}", dto.getTo(), code);

        // 置入
        redisTemplate.opsForValue().set(cacheKey, value, RedisKeyConstants.CODE_EXPIRED, TimeUnit.MILLISECONDS);

        // TODO 发送短信

        return ResultGenerator.success();
    }


    public Result<String> rebindPhone(BindPhoneDTO dto, Long userId) {
        log.info("user rebindPhone dto: {}, userId: {}", dto, userId);

        boolean f = checkCode(SendCodeEnum.REBIND_PHONE, dto.getPhone(), dto.getCode());
        if (!f) {
            throw new ServiceException(BizCodeEnum.CODE_ERROR);
        }

        User user = User.builder().id(userId).phone(dto.getPhone()).build();
        return userMapper.updateByPrimaryKeySelective(user) == 1  ? ResultGenerator.success() : ResultGenerator.fail("换绑失败");
    }


    /**
     * 验证 code
     */
    public boolean checkCode(SendCodeEnum sendCodeEnum, String to, String code) {
        String cacheKey = String.format(RedisKeyConstants.CHECK_CODE_KEY, sendCodeEnum.name(), to);
        String cacheValue = redisTemplate.opsForValue().get(cacheKey);
        if (StringUtils.isNotBlank(cacheValue)) {
            String cacheCode = cacheValue.split("_")[0];
            if (cacheCode.equalsIgnoreCase(code)) {
                // 删除验证码
                redisTemplate.delete(cacheKey);
                return true;
            }
        }

        return false;
    }


}
