package com.onsiteservice.miniapp.service.order;

import com.github.pagehelper.PageHelper;
import com.onsiteservice.common.order.vo.ServiceOrderLogVO;
import com.onsiteservice.common.order.vo.ServiceOrderVO;
import com.onsiteservice.constant.constant.BizConstants;
import com.onsiteservice.constant.enums.BizCodeEnum;
import com.onsiteservice.constant.enums.ServiceOrderOpSourceEnum;
import com.onsiteservice.constant.enums.ServiceOrderStatusEnum;
import com.onsiteservice.constant.enums.ServiceUserTypeEnum;
import com.onsiteservice.core.exception.ServiceException;
import com.onsiteservice.core.result.Result;
import com.onsiteservice.core.result.ResultGenerator;
import com.onsiteservice.dao.common.AbstractMapper;
import com.onsiteservice.dao.common.page.PageInfoVO;
import com.onsiteservice.dao.component.RecordComponent;
import com.onsiteservice.dao.mapper.service.*;
import com.onsiteservice.dao.mapper.user.UserMapper;
import com.onsiteservice.entity.address.ServiceAddress;
import com.onsiteservice.entity.category.ServiceCategory;
import com.onsiteservice.entity.category.ServiceSubclass;
import com.onsiteservice.entity.order.ServiceOrder;
import com.onsiteservice.entity.service.ServiceOrderImg;
import com.onsiteservice.entity.service.ServiceOrderLog;
import com.onsiteservice.entity.user.User;
import com.onsiteservice.miniapp.controller.order.dto.CancelServiceOrderDTO;
import com.onsiteservice.miniapp.controller.order.dto.PageServiceOrderDTO;
import com.onsiteservice.miniapp.controller.order.dto.PayServiceOrderDTO;
import com.onsiteservice.miniapp.controller.order.dto.ReserveServiceOrderDTO;
import com.onsiteservice.service.order.ServiceOrderService;
import com.onsiteservice.util.AttrCopyUtils;
import com.onsiteservice.util.RandomUtils;
import com.onsiteservice.util.aliyun.SmsUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import tk.mybatis.mapper.entity.Condition;

import javax.annotation.Resource;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;

@Service
@Slf4j
@Transactional(rollbackFor = Exception.class)
public class ServiceOrderBizService extends AbstractMapper<ServiceOrder> {


    /**
     * 短信相关
     */
    @Value("${aliyun.sms.sign}")
    private String sign;
    @Autowired
    private SmsUtils smsUtils;


    @Resource
    private ServiceOrderMapper serviceOrderMapper;
    @Resource
    private ServiceOrderImgMapper serviceOrderImgMapper;
    @Resource
    private ServiceOrderLogMapper serviceOrderLogMapper;
    @Resource
    private ServiceOrderService serviceOrderService;

    @Resource
    private ServiceAddressMapper serviceAddressMapper;

    @Resource
    private ServiceSubclassMapper serviceSubclassMapper;
    @Resource
    private ServiceCategoryMapper serviceCategoryMapper;

    @Resource
    private UserMapper userMapper;

    @Autowired
    private RecordComponent recordComponent;


    public ServiceOrderVO selectById(Long id, Long userId) {
        log.info("order selectById id: {}, userId: {}", id, userId);

        // 订单信息
        Condition c = new Condition(ServiceOrder.class);
        c.createCriteria().andEqualTo("id", id);
        List<ServiceOrder> serviceOrderList = this.selectByCondition(c);
        if (CollectionUtils.isEmpty(serviceOrderList)) {
            throw new ServiceException(BizCodeEnum.SERVICE_ORDER_NOT_EXIST);
        }

        // 基础订单信息
        ServiceOrderVO serviceOrderVO = buildServiceOrderVO(serviceOrderList).get(0);

        // 需求图片
        Condition c1 = new Condition(ServiceOrderImg.class);
        c1.createCriteria().andEqualTo(BizConstants.OrderConstants.ORDER_ID, serviceOrderVO.getId());
        List<ServiceOrderImg> serviceOrderImgList = serviceOrderImgMapper.selectByCondition(c1);
        List<String> serviceOrderImgUrlList = serviceOrderImgList.parallelStream().map(ServiceOrderImg::getUrl).collect(Collectors.toList());
        serviceOrderVO.setDemandImgUrls(serviceOrderImgUrlList);

        // 流程信息
        Condition c2 = new Condition(ServiceOrderLog.class);
        c2.createCriteria().andEqualTo(BizConstants.OrderConstants.ORDER_ID, id);
        List<ServiceOrderLog> serviceOrderLogList = serviceOrderLogMapper.selectByCondition(c2);
        List<ServiceOrderLogVO> serviceOrderLogVOList = serviceOrderLogList.parallelStream()
                .map(e -> AttrCopyUtils.copy(e, new ServiceOrderLogVO()))
                .sorted(Comparator.comparing(ServiceOrderLogVO::getCreateTime).reversed()).collect(Collectors.toList());
        serviceOrderVO.setProcess(serviceOrderLogVOList);

        return serviceOrderVO;
    }


    public Result<PageInfoVO<ServiceOrderVO>> getPage(PageServiceOrderDTO dto, Long userId) {
        log.info("order getPage dto: {}, userId: {}", dto, userId);

        PageHelper.startPage(dto.getPage(), dto.getSize());

        User user = userMapper.selectByPrimaryKey(userId);
        if (Objects.isNull(user) || !checkRoleType(user)) {
            throw new ServiceException(BizCodeEnum.NO_AUTH);
        }

        List<ServiceOrder> serviceOrderList = serviceOrderMapper.selectServiceOrderPage(dto.getOrderStatus(), dto.getMonth(), dto.getKeyWord(), user.getRoleType(), userId);

        return ResultGenerator.success(new PageInfoVO<>(buildServiceOrderVO(serviceOrderList)));
    }


    public int reserve(ReserveServiceOrderDTO dto, Long userId, ServiceOrderOpSourceEnum sourceEnum) {
        log.info("order reserve dto: {}, userId: {}", dto, userId);

        ServiceSubclass serviceSubclass = serviceSubclassMapper.selectByPrimaryKey(dto.getSubclassId());
        if (Objects.isNull(serviceSubclass)) {
            throw new ServiceException(BizCodeEnum.SERVICE_SUBCLASS_NOT_EXIST);
        }

        ServiceCategory serviceCategory = serviceCategoryMapper.selectByPrimaryKey(serviceSubclass.getCategoryId());

        ServiceAddress serviceAddress = serviceAddressMapper.selectByPrimaryKey(dto.getAddressId());
        if (Objects.isNull(serviceAddress)) {
            throw new ServiceException(BizCodeEnum.SERVICE_ADDRESS_NOT_EXIST);
        }

        ServiceOrder serviceOrder = AttrCopyUtils.copy(dto, new ServiceOrder());
        // 账户号
        serviceOrder.setAccountNo(userId);
        // 订单号
        serviceOrder.setOrderNo(RandomUtils.orderNum());
        // 客户名
        serviceOrder.setName(serviceAddress.getName());
        // 客户手机号
        serviceOrder.setPhone(serviceAddress.getPhone());
        // 客户地址
        serviceOrder.setAddress(serviceAddress.getAddress());
        // 创建人
        serviceOrder.setCreateBy(userId);
        // 服务名
        serviceOrder.setServiceName(
                String.format(BizConstants.FormatConstants.CATEGORY_SUBCLASS_SERVICE_NAME,
                        serviceCategory.getServiceName(),
                        serviceSubclass.getServiceName()));

        int result = this.insertSelective(serviceOrder);

        recordComponent.recordServiceOrderImg(serviceOrder, dto.getUrls());

        // TODO 发短信通知所有客服

        // 记录流程
        if (result == 1) {
            recordComponent.recordProcess(serviceOrder.getId(), ServiceOrderStatusEnum.RESERVE.getStatus(), ServiceOrderStatusEnum.RESERVE.getMsg(), sourceEnum, null, null, null);
        }

        return result;
    }


    public int pay(PayServiceOrderDTO dto, Long userId, ServiceOrderOpSourceEnum sourceEnum) {
        log.info("order pay dto: {}, userId: {}", dto, userId);

        // 判断能发送的状态
        ServiceOrder serviceOrder = serviceOrderService.checkOrder(dto.getId(), userId);
        if (!serviceOrder.getOrderStatus().equals(ServiceOrderStatusEnum.SEND.getStatus())) {
            throw new ServiceException(BizCodeEnum.SERVICE_ORDER_CANNOT_PAY);
        }

        serviceOrder.setOrderStatus(ServiceOrderStatusEnum.PAY.getStatus());
        serviceOrder.setModifyBy(userId);

        int result = serviceOrderMapper.updateByPrimaryKeySelective(serviceOrder);

        // TODO 调用微信支付接口

        if (result == 1) {
            recordComponent.recordProcess(serviceOrder.getId(), ServiceOrderStatusEnum.PAY.getStatus(), ServiceOrderStatusEnum.PAY.getMsg(), sourceEnum, null, null, null);
        }

        return result;
    }


    public int cancelOrder(CancelServiceOrderDTO dto, Long userId, ServiceOrderOpSourceEnum sourceEnum) {
        log.info("cancelOrder dto: {}, userId: {}", dto, userId);

        ServiceOrder serviceOrder = serviceOrderService.checkOrder(dto.getId(), userId);
        if (serviceOrder.getOrderStatus() >= ServiceOrderStatusEnum.PAY.getStatus()) {
            throw new ServiceException(BizCodeEnum.SERVICE_ORDER_CANNOT_CANCEL);
        }

        serviceOrderService.disableServiceValuatorAssign(serviceOrder, userId, null);

        serviceOrder.setOrderStatus(ServiceOrderStatusEnum.CANCEL.getStatus());
        serviceOrder.setModifyBy(userId);

        int result = serviceOrderMapper.updateByPrimaryKeySelective(serviceOrder);

        if (result == 1) {
            recordComponent.recordProcess(serviceOrder.getId(), ServiceOrderStatusEnum.CANCEL.getStatus(), ServiceOrderStatusEnum.CANCEL.getMsg(), sourceEnum, null, null, null);
        }

        return result;
    }


    /**
     * 未派单订单 用户可实时退款
     * 已派单订单 需要管理员审批
     */
    /*public int refundOrder(RefundServiceOrderDTO dto, Long userId) {
        log.info("refundOrder dto: {}, userId: {}", dto, userId);

        ServiceOrder serviceOrder = checkOrder(dto.getId(), userId);
        if (!serviceOrder.getOrderStatus().equals(ServiceOrderStatusEnum.PAY.getStatus())) {
            throw new ServiceException(BizCodeEnum.SERVICE_ORDER_CANNOT_REFUND);
        }

        // 已申请退款
        serviceOrder.setOrderStatus(ServiceOrderStatusEnum.REFUND_APPLY.getStatus());
        int result = serviceOrderMapper.updateByPrimaryKeySelective(serviceOrder);

        try {
            //  调用微信接口退款 ??

            //  通知web
        } catch (Exception e) {

        }

        return result;
    }*/


    private List<ServiceOrderVO> buildServiceOrderVO(List<ServiceOrder> serviceOrderList) {
        List<Integer> subclassIdList = serviceOrderList.parallelStream().map(ServiceOrder::getSubclassId).collect(Collectors.toList());
        List<ServiceSubclass> serviceSubclassList = serviceSubclassMapper.selectByIdList(subclassIdList);
        Map<Integer, ServiceSubclass> serviceSubclassMap = serviceSubclassList.parallelStream().collect(Collectors.toMap(ServiceSubclass::getId, Function.identity()));

        return serviceOrderList.parallelStream().map(e -> {
            ServiceOrderVO serviceOrderVO = AttrCopyUtils.copy(e, new ServiceOrderVO());
            serviceOrderVO.setSubclassImg(serviceSubclassMap.getOrDefault(e.getSubclassId(), new ServiceSubclass()).getImg());

            return serviceOrderVO;
        }).collect(Collectors.toList());
    }


    private boolean checkRoleType(User user) {
        for (ServiceUserTypeEnum userTypeEnum : ServiceUserTypeEnum.values()) {
            if (user.getRoleType().equals(userTypeEnum.getId())) {
                return true;
            }
        }

        return false;
    }


}
